create or replace TYPE generateObjectArray IS TABLE OF VARCHAR2(128);
/

create or replace PACKAGE MIGRATION_GENERATOR AS

-- Truncate All objects
PROCEDURE Truncate_MD_GENERATE;

-- Generate selected objects
PROCEDURE GenerateAll(projectID IN NUMBER, connectionID IN NUMBER, sourceDatabase IN VARCHAR2, convertVersion in NUMBER DEFAULT 0, password IN VARCHAR2);

-- Generate All Schemas
PROCEDURE GenerateSchemas(projectID IN NUMBER, connectionID IN NUMBER, sourceDatabase IN VARCHAR2, password IN VARCHAR2, schema_array IN generateObjectArray DEFAULT NULL) ;

-- Generate Tables
PROCEDURE GenerateTables(projectID IN NUMBER, connectionID IN NUMBER, sourceDatabase IN VARCHAR2, schema_array IN generateObjectArray DEFAULT NULL) ;
PROCEDURE GenerateTables(projectID IN NUMBER, connectionID IN NUMBER, sourceDatabase IN VARCHAR2, schema_name varchar2, table_array IN generateObjectArray DEFAULT NULL) ;

-- Generate Job Scheduler
PROCEDURE GenerateSchedulerJobs(starting_date IN VARCHAR2) ;

-- Generate All Tables Defaults
PROCEDURE GenerateTableDefaults(projectID IN NUMBER, connectionID IN NUMBER, sourceDatabase IN VARCHAR2, schema_array IN generateObjectArray DEFAULT NULL) ;
PROCEDURE GenerateTableDefaults(projectID IN NUMBER, connectionID IN NUMBER, sourceDatabase IN VARCHAR2, schema_name varchar2, table_array IN generateObjectArray DEFAULT NULL) ;

-- Generate Primary Keys
PROCEDURE GeneratePrimaryKeyConstraints(projectID IN NUMBER, connectionID IN NUMBER, sourceDatabase IN VARCHAR2, schema_array IN generateObjectArray DEFAULT NULL) ;
PROCEDURE GeneratePrimaryKeyConstraints(projectID IN NUMBER, connectionID IN NUMBER, sourceDatabase IN VARCHAR2, schema_name varchar2, table_array IN generateObjectArray DEFAULT NULL) ;

-- Generate Foreign Keys
PROCEDURE GenerateForeignKeyConstraints(projectID IN NUMBER, connectionID IN NUMBER, sourceDatabase IN VARCHAR2, schema_array IN generateObjectArray DEFAULT NULL) ;
PROCEDURE GenerateForeignKeyConstraints(projectID IN NUMBER, connectionID IN NUMBER, sourceDatabase IN VARCHAR2, schema_name varchar2, table_array IN generateObjectArray DEFAULT NULL) ;
PROCEDURE GenerateForeignKeyConstraints(projectID IN NUMBER, connectionID IN NUMBER, sourceDatabase IN VARCHAR2, schema_name varchar2, table_name IN varchar2, foreignKey_array IN generateObjectArray DEFAULT NULL) ;

-- Generate Unique Keys
PROCEDURE GenerateUniqueKeyConstraints(projectID IN NUMBER, connectionID IN NUMBER, sourceDatabase IN VARCHAR2, schema_array IN generateObjectArray DEFAULT NULL) ;
PROCEDURE GenerateUniqueKeyConstraints(projectID IN NUMBER, connectionID IN NUMBER, sourceDatabase IN VARCHAR2, schema_name varchar2, table_array IN generateObjectArray DEFAULT NULL) ;
PROCEDURE GenerateUniqueKeyConstraints(projectID IN NUMBER, connectionID IN NUMBER, sourceDatabase IN VARCHAR2, schema_name varchar2, table_name IN varchar2, uniqueKey_array IN generateObjectArray DEFAULT NULL) ;

-- Generate Check Constraints
PROCEDURE GenerateCheckConstraints(projectID IN NUMBER, connectionID IN NUMBER, sourceDatabase IN VARCHAR2, schema_array IN generateObjectArray DEFAULT NULL) ;
PROCEDURE GenerateCheckConstraints(projectID IN NUMBER, connectionID IN NUMBER, sourceDatabase IN VARCHAR2, schema_name varchar2, table_array IN generateObjectArray DEFAULT NULL) ;
PROCEDURE GenerateCheckConstraints(projectID IN NUMBER, connectionID IN NUMBER, sourceDatabase IN VARCHAR2, schema_name varchar2, table_name IN varchar2, checkConstraint_array IN generateObjectArray DEFAULT NULL) ;

-- Generate Unique and Non Unique Indexes
PROCEDURE GenerateIndexes(projectID IN NUMBER, connectionID IN NUMBER, sourceDatabase IN VARCHAR2, schema_array IN generateObjectArray DEFAULT NULL) ;
PROCEDURE GenerateIndexes(projectID IN NUMBER, connectionID IN NUMBER, sourceDatabase IN VARCHAR2, schema_name varchar2, table_array IN generateObjectArray DEFAULT NULL) ;
PROCEDURE GenerateIndexes(projectID IN NUMBER, connectionID IN NUMBER, sourceDatabase IN VARCHAR2, schema_name varchar2, table_name IN varchar2, index_array IN generateObjectArray DEFAULT NULL) ;

-- Generate Sequences
PROCEDURE GenerateSequences(projectID IN NUMBER, connectionID IN NUMBER, sourceDatabase IN VARCHAR2, schema_array IN generateObjectArray DEFAULT NULL) ;
PROCEDURE GenerateSequences(projectID IN NUMBER, connectionID IN NUMBER, sourceDatabase IN VARCHAR2, schema_name varchar2, sequence_array IN generateObjectArray DEFAULT NULL) ;

-- Generate GenerateTriggers
PROCEDURE GenerateTriggers(projectID IN NUMBER, connectionID IN NUMBER, sourceDatabase IN VARCHAR2, schema_array IN generateObjectArray DEFAULT NULL) ;
PROCEDURE GenerateTriggers(projectID IN NUMBER, connectionID IN NUMBER, sourceDatabase IN VARCHAR2, schema_name varchar2, trigger_array IN generateObjectArray DEFAULT NULL) ;

-- Generate Functions and Procedures
PROCEDURE GenerateStoredPrograms(projectID IN NUMBER, connectionID IN NUMBER, sourceDatabase IN VARCHAR2, schema_array IN generateObjectArray DEFAULT NULL) ;
PROCEDURE GenerateStoredPrograms(projectID IN NUMBER, connectionID IN NUMBER, sourceDatabase IN VARCHAR2, schema_name varchar2, storedProgram_array IN generateObjectArray DEFAULT NULL) ;

-- Generate Views
PROCEDURE GenerateViews(projectID IN NUMBER, connectionID IN NUMBER, sourceDatabase IN VARCHAR2, schema_array IN generateObjectArray DEFAULT NULL) ;
PROCEDURE GenerateViews(projectID IN NUMBER, connectionID IN NUMBER, sourceDatabase IN VARCHAR2, schema_name varchar2, view_array IN generateObjectArray DEFAULT NULL) ;

-- Generate Package Spec and Package Body
PROCEDURE GeneratePackages(projectID IN NUMBER, connectionID IN NUMBER, sourceDatabase IN VARCHAR2, schema_array IN generateObjectArray DEFAULT NULL) ;
PROCEDURE GeneratePackages(projectID IN NUMBER, connectionID IN NUMBER, sourceDatabase IN VARCHAR2, schema_name varchar2, package_array IN generateObjectArray DEFAULT NULL) ;

-- Generate GenerateSynonyms
PROCEDURE GenerateSynonyms(projectID IN NUMBER, connectionID IN NUMBER, sourceDatabase IN VARCHAR2, schema_array IN generateObjectArray DEFAULT NULL) ;
PROCEDURE GenerateSynonyms(projectID IN NUMBER, connectionID IN NUMBER, sourceDatabase IN VARCHAR2, schema_name varchar2, synonym_array IN generateObjectArray DEFAULT NULL) ;

-- Generate All Redshift Unload Script
-- PROCEDURE GenerateUnloadScript(projectID IN NUMBER, connectionID IN NUMBER, sourceDatabase IN VARCHAR2, unload_location_dir varchar2, my_im_role varchar2, unload_options varchar2);

-- Generate All Redshift Unload Script with aws_access_key and aws_secret_access_key
PROCEDURE GenerateUnloadScript(projectID IN NUMBER, connectionID IN NUMBER, sourceDatabase IN VARCHAR2, unload_location_dir varchar2, aws_access_key varchar2, aws_secret_access_key varchar2, unload_options varchar2);

-- Generate All External Table Script with insert into table
PROCEDURE GenerateExternalTables(projectID IN NUMBER, connectionID IN NUMBER, sourceDatabase IN VARCHAR2, datafile_dir varchar2, logfile_dir varchar2, record_delimiter varchar2, field_terminator varchar2);

-- Generate OCA external Tables
PROCEDURE GenerateOCAExternalTables(projectID IN NUMBER, connectionID IN NUMBER, sourceDatabase IN VARCHAR2, credential_name VARCHAR2, file_uri_list VARCHAR2, file_format VARCHAR2);

-- Generate OCA Copy Data
PROCEDURE GenerateOCACopyData(projectID IN NUMBER, connectionID IN NUMBER, sourceDatabase IN VARCHAR2, credential_name VARCHAR2, file_uri_list VARCHAR2, file_format VARCHAR2);

PROCEDURE GenerateOCACopyDataUtilPkg( projectID IN NUMBER, connectionID IN NUMBER, sourceDatabase IN VARCHAR2, credential_name VARCHAR2, file_uri_list VARCHAR2, file_format VARCHAR2);

-- Generate a schema
FUNCTION gen_schema_sql(projectID MD_PROJECTS.PROJECT_ID%TYPE, connectionID MD_CONNECTIONS.CONNECTION_ID%TYPE, catalogName MD_CATALOGS.CATALOG_NAME%TYPE, schemaName MD_SCHEMAS.ORACLE_NAME%TYPE, password VARCHAR2) RETURN CLOB ;

-- Generate a Table
--FUNCTION generate_table(projectID MD_PROJECTS.PROJECT_ID%TYPE, connectionID MD_CONNECTIONS.CONNECTION_ID%TYPE, catalogName MD_CATALOGS.CATALOG_NAME%TYPE, schemaName MD_SCHEMAS.ORACLE_NAME%TYPE, tableName MD_TABLES.ORACLE_NAME%TYPE) RETURN CLOB ;
FUNCTION gen_table_sql(projectID MD_PROJECTS.PROJECT_ID%TYPE, connectionID MD_CONNECTIONS.CONNECTION_ID%TYPE, catalogName MD_CATALOGS.CATALOG_NAME%TYPE, 
schemaName MD_SCHEMAS.SCHEMA_NAME%TYPE, schemaOracleName MD_SCHEMAS.ORACLE_NAME%TYPE, tableName MD_TABLES.TABLE_NAME%TYPE, tableOracleName MD_TABLES.ORACLE_NAME%TYPE) RETURN CLOB ;

-- Generate a Table Defaults
FUNCTION gen_tabledefaults_sql(projectID MD_PROJECTS.PROJECT_ID%TYPE, connectionID MD_CONNECTIONS.CONNECTION_ID%TYPE, catalogName MD_CATALOGS.CATALOG_NAME%TYPE, 
schemaName MD_SCHEMAS.SCHEMA_NAME%TYPE, schemaOracleName MD_SCHEMAS.ORACLE_NAME%TYPE, tableName MD_TABLES.TABLE_NAME%TYPE, tableOracleName MD_TABLES.ORACLE_NAME%TYPE) RETURN CLOB ;

-- Generate a Primary Key
FUNCTION gen_primaryKey_sql(projectID MD_PROJECTS.PROJECT_ID%TYPE, connectionID MD_CONNECTIONS.CONNECTION_ID%TYPE, catalogName MD_CATALOGS.CATALOG_NAME%TYPE, 
schemaName MD_SCHEMAS.SCHEMA_NAME%TYPE, schemaOracleName MD_SCHEMAS.ORACLE_NAME%TYPE, tableName MD_TABLES.TABLE_NAME%TYPE, tableOracleName MD_TABLES.ORACLE_NAME%TYPE, 
pkName MD_CONSTRAINTS.CONSTRAINT_NAME%TYPE) RETURN CLOB ;

-- Generate a Primary Key
FUNCTION gen_foreignKey_sql(projectID MD_PROJECTS.PROJECT_ID%TYPE, connectionID MD_CONNECTIONS.CONNECTION_ID%TYPE, catalogName MD_CATALOGS.CATALOG_NAME%TYPE, 
schemaName MD_SCHEMAS.SCHEMA_NAME%TYPE, schemaOracleName MD_SCHEMAS.ORACLE_NAME%TYPE, tableName MD_TABLES.TABLE_NAME%TYPE, tableOracleName MD_TABLES.ORACLE_NAME%TYPE, 
fkName MD_CONSTRAINTS.CONSTRAINT_NAME%TYPE) RETURN CLOB ;

-- Generate a Unique Key
FUNCTION gen_uniqueKey_sql(projectID MD_PROJECTS.PROJECT_ID%TYPE, connectionID MD_CONNECTIONS.CONNECTION_ID%TYPE, catalogName MD_CATALOGS.CATALOG_NAME%TYPE, 
schemaName MD_SCHEMAS.SCHEMA_NAME%TYPE, schemaOracleName MD_SCHEMAS.ORACLE_NAME%TYPE, tableName MD_TABLES.TABLE_NAME%TYPE, tableOracleName MD_TABLES.ORACLE_NAME%TYPE, 
ukName MD_CONSTRAINTS.CONSTRAINT_NAME%TYPE) RETURN CLOB ;

-- Generate a Check Constraint
FUNCTION gen_checkConstraint_sql(projectID MD_PROJECTS.PROJECT_ID%TYPE, connectionID MD_CONNECTIONS.CONNECTION_ID%TYPE, catalogName MD_CATALOGS.CATALOG_NAME%TYPE, 
schemaName MD_SCHEMAS.SCHEMA_NAME%TYPE, schemaOracleName MD_SCHEMAS.ORACLE_NAME%TYPE, tableName MD_TABLES.TABLE_NAME%TYPE, tableOracleName MD_TABLES.ORACLE_NAME%TYPE, 
checkName MD_CONSTRAINTS.CONSTRAINT_NAME%TYPE) RETURN CLOB ;

-- Generate a Indexes Key
FUNCTION gen_index_sql(projectID MD_PROJECTS.PROJECT_ID%TYPE, connectionID MD_CONNECTIONS.CONNECTION_ID%TYPE, catalogName MD_CATALOGS.CATALOG_NAME%TYPE, 
schemaName MD_SCHEMAS.SCHEMA_NAME%TYPE, schemaOracleName MD_SCHEMAS.ORACLE_NAME%TYPE, tableName MD_TABLES.TABLE_NAME%TYPE, tableOracleName MD_TABLES.ORACLE_NAME%TYPE, 
indexName MD_INDEXES.INDEX_NAME%TYPE) RETURN CLOB ;

FUNCTION gen_indextype_clause_sql(projectID MD_PROJECTS.PROJECT_ID%TYPE, connectionID MD_CONNECTIONS.CONNECTION_ID%TYPE, catalogName MD_CATALOGS.CATALOG_NAME%TYPE, 
schemaName MD_SCHEMAS.SCHEMA_NAME%TYPE, indexName MD_INDEXES.INDEX_NAME%TYPE) RETURN VARCHAR2  ;

-- Generate a Sequences
FUNCTION gen_sequence_sql(projectID MD_PROJECTS.PROJECT_ID%TYPE, connectionID MD_CONNECTIONS.CONNECTION_ID%TYPE, catalogName MD_CATALOGS.CATALOG_NAME%TYPE, 
schemaName MD_SCHEMAS.SCHEMA_NAME%TYPE, schemaOracleName MD_SCHEMAS.ORACLE_NAME%TYPE, sequenceName MD_SEQUENCES.SEQUENCE_NAME%TYPE, sequenceOracleName MD_SEQUENCES.ORACLE_NAME%TYPE) RETURN CLOB ;

-- Generate a triggers
FUNCTION gen_trigger_sql(projectID MD_PROJECTS.PROJECT_ID%TYPE, connectionID MD_CONNECTIONS.CONNECTION_ID%TYPE, catalogName MD_CATALOGS.CATALOG_NAME%TYPE, schemaName MD_SCHEMAS.ORACLE_NAME%TYPE, trigName MD_TRIGGERS.ORACLE_NAME%TYPE) RETURN CLOB ;

-- Generate a triggers
FUNCTION gen_storedprograms_sql(projectID MD_PROJECTS.PROJECT_ID%TYPE, connectionID MD_CONNECTIONS.CONNECTION_ID%TYPE, catalogName MD_CATALOGS.CATALOG_NAME%TYPE, schemaName MD_SCHEMAS.ORACLE_NAME%TYPE, sprogName MD_STORED_PROGRAMS.ORACLE_NAME%TYPE) RETURN CLOB ;

-- Generate a views
FUNCTION gen_view_sql(projectID MD_PROJECTS.PROJECT_ID%TYPE, connectionID MD_CONNECTIONS.CONNECTION_ID%TYPE, catalogName MD_CATALOGS.CATALOG_NAME%TYPE, schemaName MD_SCHEMAS.ORACLE_NAME%TYPE, viewName MD_VIEWS.ORACLE_NAME%TYPE) RETURN CLOB ;

-- Generate a Synonyms
FUNCTION gen_synonym_sql(projectID MD_PROJECTS.PROJECT_ID%TYPE, connectionID MD_CONNECTIONS.CONNECTION_ID%TYPE, catalogName MD_CATALOGS.CATALOG_NAME%TYPE, 
schemaName MD_SCHEMAS.SCHEMA_NAME%TYPE, schemaOracleName MD_SCHEMAS.ORACLE_NAME%TYPE, synonymName MD_SYNONYMS.SYNONYM_NAME%TYPE, synonymOracleName MD_SYNONYMS.ORACLE_NAME%TYPE) RETURN CLOB ;

-- Generate a Unload Script : usage of my_im_role
/*
FUNCTION generate_unload_script(projectID MD_PROJECTS.PROJECT_ID%TYPE, connectionID MD_CONNECTIONS.CONNECTION_ID%TYPE, catalogName MD_CATALOGS.CATALOG_NAME%TYPE, 
schemaName MD_SCHEMAS.SCHEMA_NAME%TYPE, schemaOracleName MD_SCHEMAS.ORACLE_NAME%TYPE, tableName MD_TABLES.TABLE_NAME%TYPE, tableOracleName MD_TABLES.ORACLE_NAME%TYPE, 
unload_location_dir varchar2, my_im_role varchar2, unload_options varchar2) RETURN CLOB ;
*/
-- Generate All Redshift Unload Script with aws_access_key and aws_secret_access_key
FUNCTION generate_unload_script(projectID MD_PROJECTS.PROJECT_ID%TYPE, connectionID MD_CONNECTIONS.CONNECTION_ID%TYPE, catalogName MD_CATALOGS.CATALOG_NAME%TYPE, 
schemaName MD_SCHEMAS.SCHEMA_NAME%TYPE, schemaOracleName MD_SCHEMAS.ORACLE_NAME%TYPE, tableName MD_TABLES.TABLE_NAME%TYPE, tableOracleName MD_TABLES.ORACLE_NAME%TYPE, 
unload_location_dir varchar2, aws_access_key varchar2, aws_secret_access_key varchar2, unload_options varchar2) RETURN CLOB ;

-- Generate a External Table
FUNCTION generate_external_table(projectID MD_PROJECTS.PROJECT_ID%TYPE, connectionID MD_CONNECTIONS.CONNECTION_ID%TYPE, catalogName MD_CATALOGS.CATALOG_NAME%TYPE, 
schemaName MD_SCHEMAS.SCHEMA_NAME%TYPE, schemaOracleName MD_SCHEMAS.ORACLE_NAME%TYPE, tableName MD_TABLES.TABLE_NAME%TYPE, tableOracleName MD_TABLES.ORACLE_NAME%TYPE, 
datafile_dir varchar2, logfile_dir varchar2, record_delimiter varchar2, field_terminator varchar2) RETURN CLOB ;

-- Generate DBMS_CLOUD.create_external_table
FUNCTION generate_OCAExternalTable(projectID MD_PROJECTS.PROJECT_ID%TYPE, connectionID MD_CONNECTIONS.CONNECTION_ID%TYPE, catalogName MD_CATALOGS.CATALOG_NAME%TYPE, 
schemaName MD_SCHEMAS.SCHEMA_NAME%TYPE, schemaOracleName MD_SCHEMAS.ORACLE_NAME%TYPE, tableName MD_TABLES.TABLE_NAME%TYPE, tableOracleName MD_TABLES.ORACLE_NAME%TYPE, credential_name VARCHAR2,
file_uri_list VARCHAR2, file_format VARCHAR2) RETURN CLOB ;

-- Generate DBMS_CLOUD.copy_data

FUNCTION generate_OCACopyData(projectID MD_PROJECTS.PROJECT_ID%TYPE, connectionID MD_CONNECTIONS.CONNECTION_ID%TYPE, catalogName MD_CATALOGS.CATALOG_NAME%TYPE, 
schemaName MD_SCHEMAS.SCHEMA_NAME%TYPE, schemaOracleName MD_SCHEMAS.ORACLE_NAME%TYPE, tableName MD_TABLES.TABLE_NAME%TYPE, tableOracleName MD_TABLES.ORACLE_NAME%TYPE, source_table_num_rows MD_TABLES.NUM_ROWS%TYPE,
captureVersion MD_GENERATE.CAPT_VERSION%TYPE, convertVersion MD_GENERATE.CONV_VERSION%TYPE, generateVersion MD_GENERATE.GENERATE_VERSION%TYPE, credential_name VARCHAR2,file_uri_list VARCHAR2, file_format VARCHAR2) RETURN CLOB ;

FUNCTION generate_OCACopyData_utilPkg(projectID MD_PROJECTS.PROJECT_ID%TYPE, connectionID MD_CONNECTIONS.CONNECTION_ID%TYPE, catalogName MD_CATALOGS.CATALOG_NAME%TYPE, 
schemaName MD_SCHEMAS.SCHEMA_NAME%TYPE, schemaOracleName MD_SCHEMAS.ORACLE_NAME%TYPE, tableName MD_TABLES.TABLE_NAME%TYPE, tableOracleName MD_TABLES.ORACLE_NAME%TYPE, 
source_table_num_rows MD_TABLES.NUM_ROWS%TYPE, captureVersion MD_GENERATE.CAPT_VERSION%TYPE, convertVersion MD_GENERATE.CONV_VERSION%TYPE, generateVersion MD_GENERATE.GENERATE_VERSION%TYPE, credential_name VARCHAR2,file_uri_list VARCHAR2, file_format VARCHAR2) RETURN CLOB ;


-- Generate Table Comments
PROCEDURE GenerateTableComments;

-- Generate selected table comment
PROCEDURE GenerateTableComments( projectID IN NUMBER, connectionID IN NUMBER, sourceDatabase IN VARCHAR2);

-- Generate table comment
FUNCTION generate_table_comments (
   projectID MD_PROJECTS.PROJECT_ID%TYPE,connectionID MD_CONNECTIONS.CONNECTION_ID%TYPE, catalogName MD_CATALOGS.CATALOG_NAME%TYPE, 
   schemaName MD_SCHEMAS.SCHEMA_NAME%TYPE, schemaOracleName MD_SCHEMAS.ORACLE_NAME%TYPE, tableName MD_TABLES.TABLE_NAME%TYPE, 
   tableOracleName MD_TABLES.ORACLE_NAME%TYPE) 
RETURN CLOB ; 


-- Generate View Comments
PROCEDURE GenerateViewComments;

-- Generate selected view comment
PROCEDURE GenerateViewComments( projectID IN NUMBER, connectionID IN NUMBER, sourceDatabase IN VARCHAR2);

-- Generate view comment
FUNCTION generate_view_comments (
   projectID MD_PROJECTS.PROJECT_ID%TYPE,connectionID MD_CONNECTIONS.CONNECTION_ID%TYPE, catalogName MD_CATALOGS.CATALOG_NAME%TYPE, 
   schemaName MD_SCHEMAS.SCHEMA_NAME%TYPE, schemaOracleName MD_SCHEMAS.ORACLE_NAME%TYPE, viewName MD_VIEWS.VIEW_NAME%TYPE, 
   viewOracleName MD_VIEWS.ORACLE_NAME%TYPE) 
RETURN CLOB ; 

-- Generate Columns Comments
PROCEDURE GenerateColumnComments;

-- Generate selected index comment
PROCEDURE GenerateColumnComments( projectID IN NUMBER, connectionID IN NUMBER, sourceDatabase IN VARCHAR2);

-- Generate column comment
FUNCTION generate_column_comments (
   projectID MD_PROJECTS.PROJECT_ID%TYPE,connectionID MD_CONNECTIONS.CONNECTION_ID%TYPE, catalogName MD_CATALOGS.CATALOG_NAME%TYPE, 
   schemaName MD_SCHEMAS.SCHEMA_NAME%TYPE, schemaOracleName MD_SCHEMAS.ORACLE_NAME%TYPE, columnName MD_COLUMNS.COLUMN_NAME%TYPE, 
   columnOracleName MD_COLUMNS.ORACLE_NAME%TYPE, tableName MD_TABLES.ORACLE_NAME%TYPE) 
RETURN CLOB ;


-- Generate Index Comments
PROCEDURE GenerateIndexComments;

-- Generate selected index comment
PROCEDURE GenerateIndexComments( projectID IN NUMBER, connectionID IN NUMBER, sourceDatabase IN VARCHAR2);

-- Generate index comment
FUNCTION generate_index_comments (
   projectID MD_PROJECTS.PROJECT_ID%TYPE,connectionID MD_CONNECTIONS.CONNECTION_ID%TYPE, catalogName MD_CATALOGS.CATALOG_NAME%TYPE, 
   schemaName MD_SCHEMAS.SCHEMA_NAME%TYPE, schemaOracleName MD_SCHEMAS.ORACLE_NAME%TYPE, indexName MD_INDEXES.INDEX_NAME%TYPE, 
   indexOracleName MD_INDEXES.ORACLE_NAME%TYPE) 
RETURN CLOB  ;

END MIGRATION_GENERATOR;
/

create or replace PACKAGE BODY MIGRATION_GENERATOR AS
a_startTime   TIMESTAMP;
g_startTime   TIMESTAMP;
g_endTime     TIMESTAMP;

g_generateVersionId NUMBER := 0;
g_convertVersionId NUMBER := 0;
g_captureVersionId NUMBER := 0;
missing_version EXCEPTION;

PROCEDURE GenerateVersionID(projectID MD_PROJECTS.PROJECT_ID%TYPE, connectionID MD_CONNECTIONS.CONNECTION_ID%TYPE, sourceDatabase IN VARCHAR2, convert_version IN NUMBER) AS

 v_conv_count NUMBER;
 v_conv_version NUMBER;
 v_capt_version NUMBER;
 v_generate_count NUMBER;

 BEGIN
    g_convertVersionId := convert_version;
	IF convert_version = 0 or convert_version IS NULL THEN

		   SELECT COUNT(CONV_VERSION) INTO v_conv_count 
		   FROM  MD_CATALOGS 
		   WHERE PROJECT_ID = projectID AND CONNECTION_ID = connectionID AND TYPE = 'CONVERTED';

		   IF v_conv_count = 0 THEN
		      RAISE missing_version ;
		   ELSIF v_conv_count > 0 THEN
			       SELECT MAX(CONV_VERSION) INTO v_conv_version 
				   FROM  MD_CATALOGS 
				   WHERE PROJECT_ID = projectID AND CONNECTION_ID = connectionID AND TYPE = 'CONVERTED';
				   IF v_conv_version = 0 THEN
				     RAISE missing_version;
				   END IF;
				   g_convertVersionId := v_conv_version;	   
		 END IF;
	ELSE
	       SELECT COUNT(CONV_VERSION) INTO v_conv_count 
		   FROM  MD_CATALOGS 
		   WHERE PROJECT_ID = projectID AND CONNECTION_ID = connectionID ;
		   g_convertVersionId := convert_version;
	END IF;

	SELECT CAPT_VERSION INTO g_captureVersionId 
    FROM  MD_CATALOGS 
    WHERE CONV_VERSION = g_convertVersionId AND PROJECT_ID = projectID AND CONNECTION_ID = connectionID AND TYPE = 'CONVERTED';
    
  	SELECT MAX(GENERATE_VERSION) INTO v_generate_count 
    FROM  MD_GENERATE
    WHERE PROJECT_ID = projectID AND CONNECTION_ID = connectionID  and CONV_VERSION = g_convertVersionId;

	g_generateVersionId := v_generate_count;

    IF g_generateVersionId IS NULL or g_generateVersionId = 0 THEN
	    g_generateVersionId := 1;
    ELSE
 	    g_generateVersionId := v_generate_count + 1; -- increase the version number by 1.
    END IF;

	DBMS_OUTPUT.PUT_LINE('Captured Version for ' || sourceDatabase || ' is ' || g_captureVersionId);
	DBMS_OUTPUT.PUT_LINE('Convert Version for ' || sourceDatabase || ' is ' || g_convertVersionId);
    DBMS_OUTPUT.PUT_LINE('Generated Version for ' || sourceDatabase || ' is ' || g_generateVersionId);

EXCEPTION
    WHEN missing_version THEN
         RAISE missing_version;
END GenerateVersionID;


-- Generate a Schema with a password
-- ToBeDone : If it is 12c DB change to : 
-- CREATE USER <mdschema.name> IDENTIFIED BY <mdschema.name> DEFAULT TABLESPACE USERS TEMPORARY TABLESPACE TEMP QUOTA UNLIMITED ON USERS;
-- GRANT CREATE SESSION, RESOURCE, CREATE VIEW, CREATE MATERIALIZED VIEW, CREATE SYNONYM TO <mdschema.name>;
FUNCTION gen_schema_sql(projectID MD_PROJECTS.PROJECT_ID%TYPE, connectionID MD_CONNECTIONS.CONNECTION_ID%TYPE, catalogName MD_CATALOGS.CATALOG_NAME%TYPE, schemaName MD_SCHEMAS.ORACLE_NAME%TYPE, password IN VARCHAR2) RETURN CLOB  IS
  clob_result CLOB;
  BEGIN
  SELECT TO_CLOB(result.script) into clob_result FROM
  (
      SELECT
        'CREATE USER '|| S.ORACLE_NAME ||' identified by '|| password || ';' || CHR(10) ||
         'GRANT CREATE SESSION, DWROLE TO ' || S.ORACLE_NAME || ';' || CHR(10) ||
         'ALTER USER ' || S.ORACLE_NAME || ' QUOTA UNLIMITED ON DATA ;' || CHR(10) script
      FROM
        MD_SCHEMAS S
      WHERE
        S.TYPE = 'CONVERTED'
        AND S.PROJECT_ID = projectID
        AND S.CONNECTION_ID = connectionID
        AND S.CATALOG_NAME = catalogName
        AND S.ORACLE_NAME = schemaName
  --      AND S.CAPT_VERSION = g_captureVersionId
        AND S.CONV_VERSION = g_convertVersionId
  ) result;

  -- DBMS_OUTPUT.PUT_LINE ('Output Schema: ' || clob_result);
  return clob_result;
  END gen_schema_sql;


-- Generate a Table
FUNCTION gen_table_sql(projectID MD_PROJECTS.PROJECT_ID%TYPE, connectionID MD_CONNECTIONS.CONNECTION_ID%TYPE, catalogName MD_CATALOGS.CATALOG_NAME%TYPE, 
schemaName MD_SCHEMAS.SCHEMA_NAME%TYPE, schemaOracleName MD_SCHEMAS.ORACLE_NAME%TYPE, tableName MD_TABLES.TABLE_NAME%TYPE, tableOracleName MD_TABLES.ORACLE_NAME%TYPE) RETURN CLOB  IS

  clob_result CLOB;
  BEGIN

  SELECT TO_CLOB(tableDdl) into clob_result FROM
  (
--    SELECT 'PROMPT Create Table '|| schemaOracleName || '.' || tableOracleName || ';' || CHR(10)||
    SELECT
           'CREATE TABLE ' || schemaOracleName || '.' || tableOracleName || CHR(10)|| '('
--         'CREATE TABLE ' || tableOracleName || CHR(10)|| '('
         || RTRIM(XMLAGG(XMLELEMENT(E, CHR(10) || '   ' ||
          C.ORACLE_NAME ||' ' ||
          C.COLUMN_TYPE ||
          DECODE(C.COLUMN_TYPE,'INT',null,'DATE',null,'TIMESTAMP',null,'TIMESTAMP WITH TIME ZONE',null,'INTERVAL DAY TO SECOND',null,'INTERVAL YEAR TO MONTH',null,
                               'LONG',null,'CLOB',null,'NCLOB',null,'SYS.ANYDATA',null,'SMALLINT',null,
                               'BLOB',null,'DECIMAL',null,'BINARY_DOUBLE',null,'LONG RAW',null,'RAW',DECODE(TO_CHAR(C.PRECISION), null,null,'('),
                               'NUMBER',DECODE(TO_CHAR(C.PRECISION),null,null,'('),'(') ||                    
          DECODE(C.COLUMN_TYPE,'INT',null,'DATE',null,'TIMESTAMP',null,'TIMESTAMP WITH TIME ZONE',null,'INTERVAL DAY TO SECOND',null,'INTERVAL YEAR TO MONTH',null,
                               'LONG',null,'CLOB',null,'NCLOB',null,'SYS.ANYDATA',null,'SMALLINT',null,
                               'BLOB',null,'DECIMAL',null,'BINARY_DOUBLE',null,'LONG RAW',null,'RAW', C.PRECISION, 'FLOAT',C.PRECISION,
                               'CHAR', C.PRECISION ||' CHAR', 'CHARACTER', C.PRECISION ||' CHAR', 'VARCHAR2',C.PRECISION ||' CHAR', 
                               'NCHAR',C.PRECISION,'NVARCHAR2',C.PRECISION,
                               'NUMBER',DECODE(TO_CHAR(C.PRECISION),null,null, TO_CHAR(C.PRECISION) || DECODE(TO_CHAR(C.SCALE), null,'',',' || TO_CHAR(C.SCALE)) ),  '******ERROR') ||
          DECODE(C.COLUMN_TYPE,'INT',null,'DATE',null,'TIMESTAMP',null,'TIMESTAMP WITH TIME ZONE',null,'INTERVAL DAY TO SECOND',null,'INTERVAL YEAR TO MONTH',null,
                               'LONG',null,'CLOB',null,'NCLOB',null,'SYS.ANYDATA',null,'SMALLINT',null,
                               'BLOB',null,'DECIMAL',null,'BINARY_DOUBLE',null,'LONG RAW',null,
                               'NUMBER',DECODE(TO_CHAR(C.PRECISION),null,null,')'),')') ||
          DECODE(nullable,'N',' NOT NULL',null) ||',') ORDER BY C.COLUMN_ORDER ).EXTRACT('/E/text()').GETCLOBVAL(), ',') ||CHR(10)|| ')' tableDdl
      FROM
          MD_COLUMNS C
      WHERE
        C.TYPE = 'CONVERTED'
        AND C.PROJECT_ID = projectID
        AND C.CONNECTION_ID =  connectionID
        AND C.CATALOG_NAME = catalogName
        AND C.SCHEMA_NAME = schemaName
        AND C.TABLE_NAME = tableName
        -- AND CAPT_VERSION = g_captureVersionId
        AND C.CONV_VERSION = g_convertVersionId
        GROUP BY C.TABLE_NAME, C.SCHEMA_NAME
      ) DUAL ;

    -- DBMS_OUTPUT.PUT_LINE ('output table: ' || clob_result);
    return clob_result;
  END gen_table_sql;
        
-- Generate a Table
-- ToBeDone : for  a blank value may need a check for charlen?
FUNCTION gen_tabledefaults_sql(projectID MD_PROJECTS.PROJECT_ID%TYPE, connectionID MD_CONNECTIONS.CONNECTION_ID%TYPE, catalogName MD_CATALOGS.CATALOG_NAME%TYPE, 
schemaName MD_SCHEMAS.SCHEMA_NAME%TYPE, schemaOracleName MD_SCHEMAS.ORACLE_NAME%TYPE, tableName MD_TABLES.TABLE_NAME%TYPE, tableOracleName MD_TABLES.ORACLE_NAME%TYPE) RETURN CLOB  IS
  clob_result CLOB;
  BEGIN

  SELECT TO_CLOB(tableDefaultDdl) into clob_result from (
    SELECT REPLACE(XMLAGG(XMLELEMENT(E,
        'ALTER TABLE ' || schemaOracleName || '.' || tableOracleName || ' MODIFY ('|| C.ORACLE_NAME || ' DEFAULT '|| C.DEFAULT_VALUE ||') ;'||CHR(10))).EXTRACT('/E/text()').GETCLOBVAL(), CHR(38)||'apos;', '''')  tableDefaultDdl
    FROM
          MD_COLUMNS C
    WHERE
        C.TYPE = 'CONVERTED'
        AND C.PROJECT_ID = projectID
        AND C.CONNECTION_ID =  connectionID
        AND C.CATALOG_NAME = catalogName
        AND C.SCHEMA_NAME = schemaName
        AND C.TABLE_NAME = tableName
        AND C.DEFAULT_VALUE is NOT NULL
        -- AND CAPT_VERSION = g_captureVersionId
        AND C.CONV_VERSION = g_convertVersionId
        GROUP BY C.TABLE_NAME
      ) DUAL ;

    -- DBMS_OUTPUT.PUT_LINE ('output table: ' || clob_result);
    return clob_result;
  END gen_tabledefaults_sql;


FUNCTION gen_primaryKey_sql(projectID MD_PROJECTS.PROJECT_ID%TYPE, connectionID MD_CONNECTIONS.CONNECTION_ID%TYPE, catalogName MD_CATALOGS.CATALOG_NAME%TYPE, 
schemaName MD_SCHEMAS.SCHEMA_NAME%TYPE, schemaOracleName MD_SCHEMAS.ORACLE_NAME%TYPE, tableName MD_TABLES.TABLE_NAME%TYPE, tableOracleName MD_TABLES.ORACLE_NAME%TYPE, 
pkName MD_CONSTRAINTS.CONSTRAINT_NAME%TYPE) RETURN CLOB  IS
  clob_result CLOB;
  BEGIN

-- DBMS_OUTPUT.PUT_LINE ('s->'||schemaName||' t->'||tableName||' pk->' ||pkName);
/*
   DBMS_OUTPUT.PUT_LINE ('projectID Name ->' || projectID);
   DBMS_OUTPUT.PUT_LINE ('connectionID Name ->' || connectionID);
   DBMS_OUTPUT.PUT_LINE ('catalogName Name ->' || catalogName);
   DBMS_OUTPUT.PUT_LINE ('schemaName Name ->' || schemaName);
   DBMS_OUTPUT.PUT_LINE ('schemaName Oracle Name ->' || schemaOracleName);

   DBMS_OUTPUT.PUT_LINE ('Table Oracle Name ->' || tableOracleName);
*/   
SELECT TO_CLOB(pkScript) into clob_result from (
SELECT 'PROMPT Creating Primary Key Constraint '|| CON.ORACLE_NAME ||';' || CHR(10)||
           'ALTER TABLE ' || schemaOracleName || '.' || tableOracleName || CHR(10)|| 'ADD CONSTRAINT ' || CON.ORACLE_NAME || ' PRIMARY KEY' || CHR(10)|| '('||
--           'ALTER TABLE ' || tableOracleName || CHR(10)|| 'ADD CONSTRAINT ' || CON.ORACLE_NAME || ' PRIMARY KEY' || CHR(10)|| '('||
RTRIM (XMLAGG (XMLELEMENT (e, CHR(10) ||'   ' || COL.ORACLE_NAME ||  ',' ) ORDER BY CON.DETAIL_ORDER ).EXTRACT ('//text()'), ',') || CHR(10) || ')' || CHR(10) || 'ENABLE ;' || CHR(10) ||
	(SELECT 'GRANT ALL ON ' || GRNT.reftable_name || ' TO ' || GRNT.schema_name || ' ;' || CHR(10) 
	 FROM 
		(
		SELECT DISTINCT C_GRNT.reftable_name, C_GRNT.schema_name
		FROM MD_CONSTRAINTS C_GRNT
		WHERE 
		C_GRNT.type = 'CONVERTED'
		AND C_GRNT.CONSTRAINT_TYPE = 'FOREIGN KEY'
		AND C_GRNT.PROJECT_ID = projectID
		AND C_GRNT.CONNECTION_ID = connectionID
		AND C_GRNT.CATALOG_NAME = catalogName
		AND C_GRNT.REFSCHEMA_NAME = schemaName
		AND C_GRNT.REFTABLE_NAME = tableName
		AND C_GRNT.SCHEMA_NAME <> C_GRNT.REFSCHEMA_NAME
		-- AND CAPT_VERSION = g_captureVersionId
        AND C_GRNT.CONV_VERSION = g_convertVersionId
		) GRNT
	)
pkScript
FROM
   MD_CONSTRAINTS CON, MD_COLUMNS COL
WHERE CON.CONSTRAINT_TYPE = 'PK'
      AND CON.TYPE = 'CONVERTED'
      AND CON.PROJECT_ID = projectID
      AND CON.CONNECTION_ID = connectionID
      AND CON.CATALOG_NAME = catalogName
      AND CON.SCHEMA_NAME = schemaName
      AND CON.TABLE_NAME = tableName
      AND CON.CONSTRAINT_NAME = pkName
      AND COL.TYPE = CON.TYPE
      AND COL.PROJECT_ID = CON.PROJECT_ID
      AND COL.CONNECTION_ID = CON.CONNECTION_ID
      AND COL.CATALOG_NAME = CON.CATALOG_NAME
      AND COL.SCHEMA_NAME = CON.SCHEMA_NAME
      AND COL.TABLE_NAME = CON.TABLE_NAME
      AND COL.COLUMN_NAME = CON.COLUMN_NAME
      -- AND CAPT_VERSION = g_captureVersionId
      AND COL.CONV_VERSION = CON.CONV_VERSION
      AND CON.CONV_VERSION = g_convertVersionId
GROUP BY
   CON.SCHEMA_NAME, CON.TABLE_NAME, CON.ORACLE_NAME
   ) dual;

   return clob_result;

    EXCEPTION 
      WHEN DUP_VAL_ON_INDEX THEN
      DBMS_OUTPUT.PUT_LINE('generate_primaryKey : Cannot insert duplicate');
      DBMS_OUTPUT.PUT_LINE('Primary Keys -> ' || pkName);
--   DBMS_OUTPUT.PUT_LINE ('Primary Keys' || clob_result);

END gen_primaryKey_sql;

-- Create Foreign Key
-- ToBeDone : Check REFTABLE_NAME is ORACLE_NAME and REFCOLUMN_NAME is ORACLE_NAME
-- Referenced Table schema should point to REFSCHEMA_NAME
FUNCTION gen_foreignKey_sql(projectID MD_PROJECTS.PROJECT_ID%TYPE, connectionID MD_CONNECTIONS.CONNECTION_ID%TYPE, catalogName MD_CATALOGS.CATALOG_NAME%TYPE, 
schemaName MD_SCHEMAS.SCHEMA_NAME%TYPE, schemaOracleName MD_SCHEMAS.ORACLE_NAME%TYPE, tableName MD_TABLES.TABLE_NAME%TYPE, tableOracleName MD_TABLES.ORACLE_NAME%TYPE, 
fkName MD_CONSTRAINTS.CONSTRAINT_NAME%TYPE) RETURN CLOB  IS

  clob_result CLOB;
  BEGIN

  -- DBMS_OUTPUT.PUT_LINE ('Constaint Name ->' || fkName);

SELECT TO_CLOB(fkScript) into clob_result FROM (
SELECT 'PROMPT Creating Foreign Key Constraint '|| C.ORACLE_NAME ||';' || CHR(10)||
           'ALTER TABLE ' || schemaOracleName || '.' || tableOracleName || CHR(10)|| 'ADD CONSTRAINT ' || C.ORACLE_NAME || ' FOREIGN KEY' || CHR(10)|| '('||
--           'ALTER TABLE ' || tableOracleName || CHR(10)|| 'ADD CONSTRAINT ' || C.ORACLE_NAME || ' FOREIGN KEY' || CHR(10)|| '('||
RTRIM (XMLAGG (XMLELEMENT (e, CHR(10) ||'   ' || 
(SELECT COL.ORACLE_NAME 
   FROM  MD_COLUMNS COL 
   WHERE 
      COL.TYPE = C.TYPE
      AND COL.PROJECT_ID = C.PROJECT_ID
      AND COL.CONNECTION_ID = C.CONNECTION_ID
      AND COL.CATALOG_NAME = C.CATALOG_NAME
      AND COL.SCHEMA_NAME = C.SCHEMA_NAME
      AND COL.TABLE_NAME = C.TABLE_NAME
      AND COL.COLUMN_NAME = C.COLUMN_NAME
      AND COL.CONV_VERSION = C.CONV_VERSION
)
||  ',' ) ORDER BY C.DETAIL_ORDER ).EXTRACT ('//text()'), ',') || CHR(10) || ')' || CHR(10) ||
 'REFERENCES ' || C.REFSCHEMA_NAME ||'.'|| C.REFTABLE_NAME || CHR(10)|| '('||
-- 'REFERENCES ' || C.REFTABLE_NAME || CHR(10)|| '('||
RTRIM (XMLAGG (XMLELEMENT (e, CHR(10) ||'   ' || 
(SELECT COL.ORACLE_NAME 
   FROM  MD_COLUMNS COL 
   WHERE 
      COL.TYPE = C.TYPE
      AND COL.PROJECT_ID = C.PROJECT_ID
      AND COL.CONNECTION_ID = C.CONNECTION_ID
      AND COL.CATALOG_NAME = C.CATALOG_NAME
      AND COL.SCHEMA_NAME = C.SCHEMA_NAME
      AND COL.TABLE_NAME = C.REFTABLE_NAME
      AND COL.COLUMN_NAME = C.REFCOLUMN_NAME 
      AND COL.CONV_VERSION = C.CONV_VERSION 
)
||  ',' ) ORDER BY C.DETAIL_ORDER ).EXTRACT ('//text()'), ',') || CHR(10) || ')' || CHR(10) ||
'ENABLE ;' fkScript
FROM 
   MD_CONSTRAINTS C
WHERE C.CONSTRAINT_TYPE = 'FOREIGN KEY'
      AND C.TYPE = 'CONVERTED'
      AND C.PROJECT_ID = projectID
      AND C.CONNECTION_ID = connectionID
      AND C.CATALOG_NAME = catalogName
      AND C.SCHEMA_NAME = schemaName
      AND C.TABLE_NAME = tableName
      AND C.CONSTRAINT_NAME = fkName
      AND C.CONV_VERSION = g_convertVersionId  
GROUP BY 
   C.SCHEMA_NAME, C.ORACLE_NAME, C.TABLE_NAME, C.REFSCHEMA_NAME, C.REFTABLE_NAME
   ) dual;

--   DBMS_OUTPUT.PUT_LINE ('Primary Keys' || clob_result);
   return clob_result;

  END gen_foreignKey_sql;

-- Generate Unique Keys
-- ToBeDone : verify the all Constraint Columns are ORACLE_NAMEs from MD_COLUMNS
FUNCTION gen_uniqueKey_sql(projectID MD_PROJECTS.PROJECT_ID%TYPE, connectionID MD_CONNECTIONS.CONNECTION_ID%TYPE, catalogName MD_CATALOGS.CATALOG_NAME%TYPE, 
schemaName MD_SCHEMAS.SCHEMA_NAME%TYPE, schemaOracleName MD_SCHEMAS.ORACLE_NAME%TYPE, tableName MD_TABLES.TABLE_NAME%TYPE, tableOracleName MD_TABLES.ORACLE_NAME%TYPE, 
ukName MD_CONSTRAINTS.CONSTRAINT_NAME%TYPE) RETURN CLOB  IS

  clob_result CLOB;
  BEGIN

--   DBMS_OUTPUT.PUT_LINE ('Constaint Name ->' || pkName);

SELECT TO_CLOB(ukScript) into clob_result FROM (
SELECT 'PROMPT Creating Unique Key Constraint '|| C.ORACLE_NAME ||';' || CHR(10)||
           'ALTER TABLE ' || schemaOracleName || '.' || tableOracleName || CHR(10)|| 'ADD CONSTRAINT ' || C.ORACLE_NAME || ' UNIQUE' || CHR(10)|| '('||
--           'ALTER TABLE ' || tableOracleName || CHR(10)|| 'ADD CONSTRAINT ' || C.ORACLE_NAME || ' UNIQUE' || CHR(10)|| '('||
RTRIM (XMLAGG (XMLELEMENT (e, CHR(10) ||'   ' || COL.ORACLE_NAME ||  ',' ) ORDER BY C.DETAIL_ORDER ).EXTRACT ('//text()'), ',') || CHR(10) || ')' || CHR(10) || 'ENABLE ;' ukScript
FROM 
   MD_CONSTRAINTS C, MD_COLUMNS COL
WHERE C.CONSTRAINT_TYPE = 'UNIQUE'
      AND C.TYPE = 'CONVERTED'
      AND C.PROJECT_ID = projectID
      AND C.CONNECTION_ID = connectionID
      AND C.CATALOG_NAME = catalogName
      AND C.SCHEMA_NAME = schemaName
      AND C.TABLE_NAME = tableName
      AND C.CONSTRAINT_NAME = ukName
      AND COL.TYPE = C.TYPE
      AND COL.PROJECT_ID = C.PROJECT_ID
      AND COL.CONNECTION_ID = C.CONNECTION_ID
      AND COL.CATALOG_NAME = C.CATALOG_NAME
      AND COL.SCHEMA_NAME = C.SCHEMA_NAME
      AND COL.TABLE_NAME = C.TABLE_NAME
      AND COL.COLUMN_NAME = C.COLUMN_NAME
      AND COL.CONV_VERSION = C.CONV_VERSION
      AND C.CONV_VERSION = g_convertVersionId
GROUP BY 
   C.ORACLE_NAME, C.TABLE_NAME
   ) dual;

 --  DBMS_OUTPUT.PUT_LINE ('Unique Keys' || clob_result);
   return clob_result;

END gen_uniqueKey_sql;

-- Generate Check Constraints
-- ToBeDone : verify the all Constraint Columns are ORACLE_NAMEs from MD_COLUMNS
FUNCTION gen_checkConstraint_sql(projectID MD_PROJECTS.PROJECT_ID%TYPE, connectionID MD_CONNECTIONS.CONNECTION_ID%TYPE, catalogName MD_CATALOGS.CATALOG_NAME%TYPE, 
schemaName MD_SCHEMAS.SCHEMA_NAME%TYPE, schemaOracleName MD_SCHEMAS.ORACLE_NAME%TYPE, tableName MD_TABLES.TABLE_NAME%TYPE, tableOracleName MD_TABLES.ORACLE_NAME%TYPE, 
checkName MD_CONSTRAINTS.CONSTRAINT_NAME%TYPE) RETURN CLOB  IS

  clob_result CLOB;
  BEGIN

--   DBMS_OUTPUT.PUT_LINE ('Constraint Name ->' || checkName);

SELECT TO_CLOB(checkScript) into clob_result FROM (
SELECT 'PROMPT Creating Check Constraint '|| C.ORACLE_NAME ||';' || CHR(10)||
           'ALTER TABLE ' || schemaOracleName || '.' || tableOracleName || CHR(10)|| 'ADD CONSTRAINT ' || C.ORACLE_NAME || ' CHECK ' || CHR(10)||
--           'ALTER TABLE ' || tableOracleName || CHR(10)|| 'ADD CONSTRAINT ' || C.ORACLE_NAME || ' CHECK ' || CHR(10)|| '(' ||CHR(10)||
 C.CONSTRAINT_TEXT || CHR(10) || ')' || CHR(10) || 'ENABLE ;' checkScript
FROM 
   MD_CONSTRAINTS C
WHERE C.CONSTRAINT_TYPE = 'CHECK'
      AND C.TYPE = 'CONVERTED'
      AND C.PROJECT_ID = projectID
      AND C.CONNECTION_ID = connectionID
      AND C.CATALOG_NAME = catalogName
      AND C.SCHEMA_NAME = schemaName
      AND C.TABLE_NAME = tableName
      AND C.CONSTRAINT_NAME = checkName
      AND C.CONV_VERSION = g_convertVersionId
   ) dual;

 --  DBMS_OUTPUT.PUT_LINE ('Check Constraint ' || clob_result);
   return clob_result;

END gen_checkConstraint_sql;


FUNCTION gen_index_sql(projectID MD_PROJECTS.PROJECT_ID%TYPE, connectionID MD_CONNECTIONS.CONNECTION_ID%TYPE, catalogName MD_CATALOGS.CATALOG_NAME%TYPE, 
schemaName MD_SCHEMAS.SCHEMA_NAME%TYPE, schemaOracleName MD_SCHEMAS.ORACLE_NAME%TYPE, tableName MD_TABLES.TABLE_NAME%TYPE, tableOracleName MD_TABLES.ORACLE_NAME%TYPE, 
indexName MD_INDEXES.INDEX_NAME%TYPE) RETURN CLOB  IS

  clob_result CLOB;
  BEGIN

SELECT TO_CLOB(indxScript || gen_indextype_clause_sql( projectID, connectionID, catalogName, schemaName, indexName) || ';') INTO clob_result FROM (
SELECT 'PROMPT Creating Index '|| I.ORACLE_NAME ||';' || CHR(10)||
           'CREATE '|| DECODE(I.INDEX_TYPE, 'NON_UNIQUE', null, 'UNIQUE', 'UNIQUE ') ||'INDEX ' || schemaOracleName ||'.'|| I.ORACLE_NAME ||' ON '|| tableOracleName || CHR(10)|| '('||
--           'CREATE '|| DECODE(I.INDEX_TYPE, 'NON_UNIQUE', null, 'UNIQUE', 'UNIQUE ') ||'INDEX ' || I.ORACLE_NAME ||' ON '|| tableOracleName || CHR(10)|| '('||
RTRIM (XMLAGG (XMLELEMENT (e, CHR(10) ||'   ' || COL.ORACLE_NAME ||  ',' ) ORDER BY I.DETAIL_ORDER ).EXTRACT ('//text()'), ',') || CHR(10) || ')' indxScript
FROM 
   MD_INDEXES I, MD_COLUMNS COL
WHERE  I.TYPE = 'CONVERTED'
      AND I.PROJECT_ID = projectID
      AND I.CONNECTION_ID = connectionID
      AND I.CATALOG_NAME = catalogName
      AND I.SCHEMA_NAME = schemaName
      AND I.TABLE_NAME = tableName
      AND I.INDEX_NAME = indexName
      AND COL.TYPE = I.TYPE
      AND COL.PROJECT_ID = I.PROJECT_ID
      AND COL.CONNECTION_ID = I.CONNECTION_ID
      AND COL.CATALOG_NAME = I.CATALOG_NAME
      AND COL.SCHEMA_NAME = I.SCHEMA_NAME
      AND COL.TABLE_NAME = I.TABLE_NAME
      AND COL.COLUMN_NAME = I.COLUMN_NAME 
      AND COL.CONV_VERSION = I.CONV_VERSION
      AND I.CONV_VERSION = g_convertVersionId 
GROUP BY 
   I.SCHEMA_NAME, I.ORACLE_NAME, I.TABLE_NAME, I.INDEX_TYPE 
   ) dual;

--   DBMS_OUTPUT.PUT_LINE ('Primary Keys' || clob_result);
   return clob_result;

END gen_index_sql;


-- Constatns for TEXT INDEX TYPES : BLOBs and CLOBs
FUNCTION gen_indextype_clause_sql(projectID MD_PROJECTS.PROJECT_ID%TYPE, connectionID MD_CONNECTIONS.CONNECTION_ID%TYPE, catalogName MD_CATALOGS.CATALOG_NAME%TYPE, 
schemaName MD_SCHEMAS.SCHEMA_NAME%TYPE, indexName MD_INDEXES.INDEX_NAME%TYPE) RETURN VARCHAR2  IS

  clause_result VARCHAR2(100);
  BEGIN
--  DBMS_OUTPUT.PUT_LINE ('s->'||schemaName||' i-> '||indexName);
 SELECT  DECODE(count_i, 0, ' ', 1, ' INDEXTYPE is ctxsys.context ', ' INDEXTYPE is ctxsys.ctxcat ') INTO clause_result
 FROM 
 (
  SELECT COUNT(*) count_i 
  FROM 
      MD_COLUMNS C, MD_INDEXES I
  WHERE
        I.TYPE = 'CONVERTED'
        AND C.TYPE = I.TYPE
        AND I.PROJECT_ID = projectID
        AND I.CONNECTION_ID = connectionID
        AND I.CATALOG_NAME = catalogName
        AND I.SCHEMA_NAME = schemaName
        AND I.INDEX_NAME = indexName
        AND C.PROJECT_ID = I.PROJECT_ID 
        AND C.CONNECTION_ID = I.CONNECTION_ID
        AND C.CATALOG_NAME = I.CATALOG_NAME
        AND C.SCHEMA_NAME = I.SCHEMA_NAME
        AND C.TABLE_NAME = I.TABLE_NAME
        AND C.COLUMN_NAME = I.COLUMN_NAME
        AND C.COLUMN_TYPE IN ('BLOB', 'CLOB')
        AND C.CONV_VERSION = I.CONV_VERSION
        AND I.CONV_VERSION = g_convertVersionId 
  GROUP BY 
        I.INDEX_NAME 
  );

--   DBMS_OUTPUT.PUT_LINE ('Index clause ' || clause_result);
   return clause_result;

END gen_indextype_clause_sql;

-- Generate Sequences
FUNCTION gen_sequence_sql(projectID MD_PROJECTS.PROJECT_ID%TYPE, connectionID MD_CONNECTIONS.CONNECTION_ID%TYPE, catalogName MD_CATALOGS.CATALOG_NAME%TYPE, 
schemaName MD_SCHEMAS.SCHEMA_NAME%TYPE, schemaOracleName MD_SCHEMAS.ORACLE_NAME%TYPE, sequenceName MD_SEQUENCES.SEQUENCE_NAME%TYPE, sequenceOracleName MD_SEQUENCES.ORACLE_NAME%TYPE) RETURN CLOB  IS

  clob_result CLOB;
  BEGIN

SELECT TO_CLOB(SeqScript) into clob_result FROM (
SELECT 'PROMPT Creating Sequence '|| sequenceOracleName ||';' || CHR(10)||
--             'CREATE '|| 'SEQUENCE ' || sequenceOracleName ||CHR(10)|| 
           'CREATE '|| 'SEQUENCE ' || schemaOracleName ||'.'|| sequenceOracleName ||CHR(10)|| 
RTRIM (XMLAGG (XMLELEMENT (e, '   ' || DECODE(S.SEQ_START, null, null, 'START WITH '|| S.SEQ_START) || CHR(10) ||
                                       DECODE(S.INCR, null, null, '   INCREMENT BY '|| S.INCR) || CHR(10) ||
                                       DECODE(S.MAX, null, '   NOMAXVALUE', ' MAXVALUE '|| S.MAX) || CHR(10) ||
                                       DECODE(S.MIN, null, '   NOMINVALUE', 'MINVALUE '|| S.MIN) || CHR(10) ||
                                       DECODE(S.CACHE, 0,  '   NOCACHE ', null, '   NOCACHE', 1, ' CACHE '|| S.CACHE_SIZE) || CHR(10) ||
                                       DECODE(S."ORDER", 0,  '   NOORDER ', null, '   NOORDER', 1, ' ORDER')
      )).EXTRACT ('//text()'), ',') || CHR(10) || ';' SeqScript
FROM 
   MD_SEQUENCES S
WHERE  S.TYPE = 'CONVERTED'
      AND S.PROJECT_ID = projectID
      AND S.CONNECTION_ID = connectionID
      AND S.CATALOG_NAME = catalogName
      AND S.SCHEMA_NAME = schemaName
      AND S.SEQUENCE_NAME = sequenceName
      AND S.CONV_VERSION = g_convertVersionId 
GROUP BY 
   S.SCHEMA_NAME, S.ORACLE_NAME
   ) dual;

--   DBMS_OUTPUT.PUT_LINE ('Primary Keys' || clob_result);
   return clob_result;

END gen_sequence_sql;

-- generate Triggers
-- TODO : need to put Prompt and post appending ; and / if required
-- ; and / can be done during translation process
FUNCTION gen_trigger_sql(projectID MD_PROJECTS.PROJECT_ID%TYPE, connectionID MD_CONNECTIONS.CONNECTION_ID%TYPE, catalogName MD_CATALOGS.CATALOG_NAME%TYPE, schemaName MD_SCHEMAS.ORACLE_NAME%TYPE, trigName MD_TRIGGERS.ORACLE_NAME%TYPE) RETURN CLOB  IS
  clob_result CLOB;
  BEGIN

SELECT T.NATIVE_SQL into clob_result FROM 
   MD_TRIGGERS T
WHERE  T.TYPE = 'CONVERTED'
      AND T.PROJECT_ID = projectID
      AND T.CONNECTION_ID = connectionID
      AND T.CATALOG_NAME = catalogName
      AND T.SCHEMA_NAME = schemaName
      AND T.ORACLE_NAME = trigName
      AND T.CONV_VERSION = g_convertVersionId 
;

--   DBMS_OUTPUT.PUT_LINE ('Primary Keys' || clob_result);
   return clob_result;

END gen_trigger_sql;


-- Generate a Functions and procedures
-- TODO : need to put Prompt and post appending ; and / if required
-- ; and / can be done during translation process
FUNCTION gen_storedprograms_sql(projectID MD_PROJECTS.PROJECT_ID%TYPE, connectionID MD_CONNECTIONS.CONNECTION_ID%TYPE, catalogName MD_CATALOGS.CATALOG_NAME%TYPE, schemaName MD_SCHEMAS.ORACLE_NAME%TYPE, sprogName MD_STORED_PROGRAMS.ORACLE_NAME%TYPE) RETURN CLOB  IS
  clob_result CLOB;
  BEGIN

SELECT T.NATIVE_SQL into clob_result FROM 
   MD_STORED_PROGRAMS T
WHERE  T.TYPE = 'CONVERTED'
      AND T.PROJECT_ID = projectID
      AND T.CONNECTION_ID = connectionID
      AND T.CATALOG_NAME = catalogName
      AND T.SCHEMA_NAME = schemaName
      AND T.ORACLE_NAME = sprogName
      AND T.CONV_VERSION = g_convertVersionId 
;
--   DBMS_OUTPUT.PUT_LINE ('Primary Keys' || clob_result);
   return clob_result;

END gen_storedprograms_sql;


-- Generate a Views
-- TODO : need to put Prompt and post appending ; and / if required
-- ; and / can be done during translation process
FUNCTION gen_view_sql(projectID MD_PROJECTS.PROJECT_ID%TYPE, connectionID MD_CONNECTIONS.CONNECTION_ID%TYPE, catalogName MD_CATALOGS.CATALOG_NAME%TYPE, schemaName MD_SCHEMAS.ORACLE_NAME%TYPE, viewName MD_VIEWS.ORACLE_NAME%TYPE) RETURN CLOB  IS
  clob_result CLOB;
  BEGIN

SELECT V.NATIVE_SQL into clob_result FROM 
   MD_VIEWS V
WHERE  V.TYPE = 'CONVERTED'
      AND V.PROJECT_ID = projectID
      AND V.CONNECTION_ID = connectionID
      AND V.CATALOG_NAME = catalogName
      AND V.SCHEMA_NAME = schemaName
      AND V.ORACLE_NAME = viewName
      AND V.CONV_VERSION = g_convertVersionId 
;
--   DBMS_OUTPUT.PUT_LINE ('Primary Keys' || clob_result);
   return clob_result;

END gen_view_sql;

-- Generate Synonyms
-- CREATE [OR REPLACE] [PUBLIC] SYNONYM [schema .] synonym_name FOR [schema .] object_name ;
FUNCTION gen_synonym_sql(projectID MD_PROJECTS.PROJECT_ID%TYPE, connectionID MD_CONNECTIONS.CONNECTION_ID%TYPE, catalogName MD_CATALOGS.CATALOG_NAME%TYPE, 
schemaName MD_SCHEMAS.SCHEMA_NAME%TYPE, schemaOracleName MD_SCHEMAS.ORACLE_NAME%TYPE, synonymName MD_SYNONYMS.SYNONYM_NAME%TYPE, synonymOracleName MD_SYNONYMS.ORACLE_NAME%TYPE) RETURN CLOB  IS

  clob_result CLOB;
  BEGIN

SELECT TO_CLOB(SymScript) into clob_result FROM (
SELECT 'PROMPT Creating Synonym '|| synonymOracleName ||';' || CHR(10)||
           'CREATE OR REPLACE '|| 
            CASE 
                WHEN S.PRIVATE_VISIBILITY= 'N'
                      THEN 'PUBLIC'
                END ||
           'SYNONYM ' || schemaOracleName ||'.'|| synonymOracleName  || ' FOR ' || S.SYNONYM_FOR_OBJECT_NAME || ';' SymScript
--           'SYNONYM ' || synonymOracleName  || ' FOR ' || S.SYNONYM_FOR_OBJECT_NAME || ';' SymScript
FROM 
   MD_SYNONYMS S
WHERE  S.TYPE = 'CONVERTED'
      AND S.PROJECT_ID = projectID
      AND S.CONNECTION_ID = connectionID
      AND S.CATALOG_NAME = catalogName
      AND S.SCHEMA_NAME = schemaName
      AND S.SYNONYM_NAME = synonymName
      AND S.CONV_VERSION = g_convertVersionId 
   ) dual;

--   DBMS_OUTPUT.PUT_LINE ('Synonyms' || clob_result);
   return clob_result;

END gen_synonym_sql;

-- Generate Unload Scripts
/*
FUNCTION generate_unload_script(projectID MD_PROJECTS.PROJECT_ID%TYPE, connectionID MD_CONNECTIONS.CONNECTION_ID%TYPE, catalogName MD_CATALOGS.CATALOG_NAME%TYPE, 
schemaName MD_SCHEMAS.SCHEMA_NAME%TYPE, schemaOracleName MD_SCHEMAS.ORACLE_NAME%TYPE, tableName MD_TABLES.TABLE_NAME%TYPE, tableOracleName MD_TABLES.ORACLE_NAME%TYPE, 
unload_location_dir varchar2, my_im_role varchar2, unload_options varchar2) RETURN CLOB  IS

   clob_result CLOB;
  BEGIN

  SELECT TO_CLOB(tableDdl) into clob_result FROM (
   SELECT 'unload (''select * from ' || schemaName || '.' || tableName || ''' )' || 
          ' to '|| '''' || unload_location_dir || '/' || schemaOracleName || '_' || tableOracleName || '''' || 
          ' credentials ' || '''' || my_im_role || '''' || ' ' || NVL2(unload_options,  unload_options,  '') || 
          ' ;' tableDdl
      FROM
          MD_TABLES T
      WHERE
        T.TYPE = 'CONVERTED'
        AND T.PROJECT_ID = projectID
        AND T.CONNECTION_ID =  connectionID
        AND T.CATALOG_NAME = catalogName
        AND T.SCHEMA_NAME = schemaName
        AND T.TABLE_NAME = tableName
        AND T.CONV_VERSION = g_convertVersionId 
        GROUP BY T.TABLE_NAME, T.SCHEMA_NAME
      ) DUAL ;

--   DBMS_OUTPUT.PUT_LINE ('Synonyms' || clob_result);
   return clob_result;

END generate_unload_script;
*/
-- unload ('select * from public.listing') to 's3://sqldev1/delimitertest/public__listing_x1' 
-- credentials 'aws_access_key_id=AKIAJJFCY433P6UFYTKQ;aws_secret_access_key=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' 
-- parallel on gzip;
FUNCTION generate_unload_script(projectID MD_PROJECTS.PROJECT_ID%TYPE, connectionID MD_CONNECTIONS.CONNECTION_ID%TYPE, catalogName MD_CATALOGS.CATALOG_NAME%TYPE, 
schemaName MD_SCHEMAS.SCHEMA_NAME%TYPE, schemaOracleName MD_SCHEMAS.ORACLE_NAME%TYPE, tableName MD_TABLES.TABLE_NAME%TYPE, tableOracleName MD_TABLES.ORACLE_NAME%TYPE, 
unload_location_dir varchar2, aws_access_key varchar2, aws_secret_access_key varchar2, unload_options varchar2) RETURN CLOB IS

   clob_result CLOB;
  BEGIN

  SELECT TO_CLOB(tableDdl) into clob_result FROM (
   SELECT 'unload (''select * from ' || schemaName || '.\"' || tableName || '\"'' )' || 
          ' to '|| '''' || unload_location_dir || '/' || schemaOracleName || '/' || tableOracleName || '/' ||'''' ||           
          ' credentials ' || '''' || 'aws_access_key_id=' || aws_access_key || ';aws_secret_access_key=' || aws_secret_access_key || '''' || ' ' || NVL2(unload_options,  unload_options,  '') || 
          ' ;' tableDdl
      FROM
          MD_TABLES T
      WHERE
        T.TYPE = 'CONVERTED'
        AND T.PROJECT_ID = projectID
        AND T.CONNECTION_ID =  connectionID
        AND T.CATALOG_NAME = catalogName
        AND T.SCHEMA_NAME = schemaName
        AND T.TABLE_NAME = tableName
        AND T.CONV_VERSION = g_convertVersionId 
        GROUP BY T.TABLE_NAME, T.SCHEMA_NAME
      ) DUAL ;

--   DBMS_OUTPUT.PUT_LINE ('Synonyms' || clob_result);
   return clob_result;

END generate_unload_script;

-- Generates External Table
FUNCTION generate_external_table(projectID MD_PROJECTS.PROJECT_ID%TYPE, connectionID MD_CONNECTIONS.CONNECTION_ID%TYPE, catalogName MD_CATALOGS.CATALOG_NAME%TYPE, 
schemaName MD_SCHEMAS.SCHEMA_NAME%TYPE, schemaOracleName MD_SCHEMAS.ORACLE_NAME%TYPE, tableName MD_TABLES.TABLE_NAME%TYPE, tableOracleName MD_TABLES.ORACLE_NAME%TYPE, 
datafile_dir varchar2, logfile_dir varchar2, record_delimiter varchar2, field_terminator varchar2) RETURN CLOB  IS

  clob_result CLOB;
  BEGIN

SELECT TO_CLOB(externaltblDdl) into clob_result FROM 
(
SELECT 'PROMPT Create External Table '|| schemaOracleName || '.' || tableOracleName || '_ext' || CHR(10)||
        'CREATE TABLE ' || schemaOracleName || '.' || tableOracleName || '_ext' || CHR(10)|| '('
         || RTRIM(XMLAGG(XMLELEMENT(E, CHR(10) || '    ' ||
(
SELECT 
          COL.ORACLE_NAME ||' ' ||
          COL.COLUMN_TYPE ||
          DECODE(COL.COLUMN_TYPE,'INT',null,'DATE',null,'TIMESTAMP',null,'TIMESTAMP WITH TIME ZONE',null,'INTERVAL DAY TO SECOND',null,'INTERVAL YEAR TO MONTH',null,
                               'LONG',null,'CLOB',null,'NCLOB',null,'SYS.ANYDATA',null,'SMALLINT',null,
                               'BLOB',null,'DECIMAL',null,'BINARY_DOUBLE',null,'LONG RAW',null,'RAW',DECODE(TO_CHAR(COL.PRECISION), null,null,'('),
                               'NUMBER',DECODE(TO_CHAR(COL.PRECISION),null,null,'('),'(') ||                    
          DECODE(COL.COLUMN_TYPE,'INT',null,'DATE',null,'TIMESTAMP',null,'TIMESTAMP WITH TIME ZONE',null,'INTERVAL DAY TO SECOND',null,'INTERVAL YEAR TO MONTH',null,
                               'LONG',null,'CLOB',null,'NCLOB',null,'SYS.ANYDATA',null,'SMALLINT',null,
                               'BLOB',null,'DECIMAL',null,'BINARY_DOUBLE',null,'LONG RAW',null,'RAW', COL.PRECISION, 'FLOAT',C.PRECISION,
                               'CHAR', COL.PRECISION ||' CHAR', 'CHARACTER', COL.PRECISION ||' CHAR', 'VARCHAR2',COL.PRECISION ||' CHAR', 
                               'NCHAR',COL.PRECISION,'NVARCHAR2',COL.PRECISION,
                               'NUMBER',DECODE(TO_CHAR(COL.PRECISION),null,null, TO_CHAR(COL.PRECISION) || DECODE(TO_CHAR(COL.SCALE), null,'',',' || TO_CHAR(COL.SCALE)) ),  '******ERROR') ||
          DECODE(COL.COLUMN_TYPE,'INT',null,'DATE',null,'TIMESTAMP',null,'TIMESTAMP WITH TIME ZONE',null,'INTERVAL DAY TO SECOND',null,'INTERVAL YEAR TO MONTH',null,
                               'LONG',null,'CLOB',null,'NCLOB',null,'SYS.ANYDATA',null,'SMALLINT',null,
                               'BLOB',null,'DECIMAL',null,'BINARY_DOUBLE',null,'LONG RAW',null,
                               'NUMBER',DECODE(TO_CHAR(COL.PRECISION),null,null,')'),')')                           
      FROM
          MD_COLUMNS COL
      WHERE
        COL.TYPE = 'CONVERTED'
        AND COL.PROJECT_ID = C.PROJECT_ID
        AND COL.CONNECTION_ID = C.CONNECTION_ID
        AND COL.CATALOG_NAME = C.CATALOG_NAME
        AND COL.SCHEMA_NAME = C.SCHEMA_NAME
        AND COL.TABLE_NAME =  C.TABLE_NAME
        AND COL.COLUMN_NAME =  C.COLUMN_NAME
        AND COL.CONV_VERSION = C.CONV_VERSION 
)
||  ',' ) ORDER BY C.COLUMN_ORDER ).EXTRACT ('//text()'), ',') ||CHR(10)|| ')' || CHR(10) ||
 'ORGANIZATION EXTERNAL' || CHR(10) ||
 '( TYPE ORACLE_LOADER' || CHR(10) ||
 '     DEFAULT DIRECTORY '|| datafile_dir || CHR(10) ||
 '     ACCESS PARAMETERS' || CHR(10) ||
 '     ( RECORDS DELIMITED BY '||''''||record_delimiter||''''|| CHR(10) ||
 '       BADFILE ' || logfile_dir ||':'||''''|| schemaOracleName || '.' || tableOracleName ||'.bad'||'''' || CHR(10) ||
 '       LOGFILE ' || logfile_dir ||':'||''''|| schemaOracleName || '.' || tableOracleName ||'.log'||'''' || CHR(10) ||
 '       FIELDS TERMINATED BY '||''''||field_terminator||''''|| CHR(10) ||
 '       NOTRIM' || CHR(10) ||
 '       MISSING FIELD VALUES ARE NULL' || CHR(10) ||
 '       REJECT ROWS WITH ALL NULL' || CHR(10) ||
 '       FIELDS' || CHR(10) ||
 '       (' ||
RTRIM (XMLAGG (XMLELEMENT (x, CHR(10) ||'       ' || '    ' ||
(
SELECT 
          CL.ORACLE_NAME || 
          DECODE(CL.COLUMN_TYPE,'DATE',' CHAR date_format TIMESTAMP mask "MON DD YYYY hh:mi:ss:FF3AM"', '')                          
      FROM
          MD_COLUMNS CL
      WHERE
        CL.TYPE = 'CONVERTED'
        AND CL.PROJECT_ID = C.PROJECT_ID
        AND CL.CONNECTION_ID = C.CONNECTION_ID
        AND CL.CATALOG_NAME = C.CATALOG_NAME
        AND CL.SCHEMA_NAME = C.SCHEMA_NAME
        AND CL.TABLE_NAME =  C.TABLE_NAME
        AND CL.COLUMN_NAME =  C.COLUMN_NAME
        AND CL.CONV_VERSION = C.CONV_VERSION 
)
||  ',' ) ORDER BY C.COLUMN_ORDER ).EXTRACT ('//text()'), ',') || CHR(10) || 
'       )' || CHR(10) || 
'     )' || CHR(10) || 
'     LOCATION ('|| ''''|| schemaOracleName || '_' || tableOracleName ||'*'||'''' ||')'|| CHR(10) ||
') REJECT LIMIT UNLIMITED ;' || CHR(10) || CHR(10) || CHR(10) ||
'INSERT INTO ' || schemaOracleName || '.' || tableOracleName || '(SELECT * FROM ' || schemaOracleName || '.' || tableOracleName || '_ext' || ');' || CHR(10) ||
'COMMIT;' || CHR(10) || CHR(10) ||
'-- DROP TABLE ' || schemaOracleName || '.' || tableOracleName || '_ext' || ';' || CHR(10)
externaltblDdl
FROM 
          MD_COLUMNS C, MD_TABLES T
      WHERE
        C.TYPE = 'CONVERTED'
        AND T.TYPE =  C.TYPE
        AND C.PROJECT_ID = projectID
        AND C.CONNECTION_ID = connectionID
        AND C.CATALOG_NAME = catalogName
        AND C.SCHEMA_NAME = schemaName
        AND C.TABLE_NAME =  tableName
        AND T.PROJECT_ID = C.PROJECT_ID
        AND T.CONNECTION_ID = C.CONNECTION_ID
        AND T.CATALOG_NAME = C.CATALOG_NAME
        AND T.SCHEMA_NAME = C.SCHEMA_NAME
        AND T.TABLE_NAME =  C.TABLE_NAME
        AND C.CONV_VERSION = T.CONV_VERSION 
        AND T.CONV_VERSION = g_convertVersionId 
GROUP BY C.CATALOG_NAME, C.SCHEMA_NAME, C.TABLE_NAME
   ) DUAL;

    -- DBMS_OUTPUT.PUT_LINE ('output table: ' || clob_result);
    return clob_result;
  END generate_external_table;


-- Generate an OCA External Table
FUNCTION generate_OCAExternalTable( 
   projectID MD_PROJECTS.PROJECT_ID%TYPE, 
   connectionID MD_CONNECTIONS.CONNECTION_ID%TYPE, 
   catalogName MD_CATALOGS.CATALOG_NAME%TYPE, 
   schemaName MD_SCHEMAS.SCHEMA_NAME%TYPE, 
   schemaOracleName MD_SCHEMAS.ORACLE_NAME%TYPE, 
   tableName MD_TABLES.TABLE_NAME%TYPE, 
   tableOracleName MD_TABLES.ORACLE_NAME%TYPE,
   credential_name VARCHAR2,
   file_uri_list VARCHAR2,
   file_format VARCHAR2
   ) RETURN CLOB  IS

  clob_result CLOB;

  BEGIN

  SELECT TO_CLOB(externalTableDdl) into clob_result FROM
  (
    SELECT 
         'DBMS_CLOUD.create_external_table (' || CHR(10)  
               || '  table_name => '''  ||  C.SCHEMA_NAME||'_' || C.TABLE_NAME ||'_ext'', ' || CHR(10) 
               || '  credential_name => '''  ||  credential_name ||''', '|| CHR(10) 
            --  || '  file_uri_list => '''  ||  file_uri_list ||'/' || C.SCHEMA_NAME||'_' ||C.TABLE_NAME || '000' ||''', ' || CHR(10) 
               || '  file_uri_list => '''  ||  file_uri_list ||'/' || schemaOracleName||'_'|| tableOracleName || '000.gz' ||''', ' || CHR(10)
               || '  field_list => '''
               || LTRIM(RTRIM(XMLAGG(XMLELEMENT(E, ' ' ||
                   C.ORACLE_NAME ||' ' ||
                   C.COLUMN_TYPE ||
                   DECODE(C.COLUMN_TYPE,'INT',null,'DATE',null,'TIMESTAMP',null,'TIMESTAMP WITH TIME ZONE',null,'INTERVAL DAY TO SECOND',null,'INTERVAL YEAR TO MONTH',null,
                               'LONG',null,'CLOB',null,'NCLOB',null,'SYS.ANYDATA',null,'SMALLINT',null,
                               'BLOB',null,'DECIMAL',null,'BINARY_DOUBLE',null,'LONG RAW',null,'RAW',DECODE(TO_CHAR(C.PRECISION), null,null,'('),
                               'NUMBER',DECODE(TO_CHAR(C.PRECISION),null,null,'('),'(') ||                    
                   DECODE(C.COLUMN_TYPE,'INT',null,'DATE',null,'TIMESTAMP',null,'TIMESTAMP WITH TIME ZONE',null,'INTERVAL DAY TO SECOND',null,'INTERVAL YEAR TO MONTH',null,
                               'LONG',null,'CLOB',null,'NCLOB',null,'SYS.ANYDATA',null,'SMALLINT',null,
                               'BLOB',null,'DECIMAL',null,'BINARY_DOUBLE',null,'LONG RAW',null,'RAW', C.PRECISION, 'FLOAT',C.PRECISION,
                               'CHAR', C.PRECISION ||' CHAR', 'CHARACTER', C.PRECISION ||' CHAR', 'VARCHAR2',C.PRECISION ||' CHAR', 
                               'NCHAR',C.PRECISION,'NVARCHAR2',C.PRECISION,
                               'NUMBER',DECODE(TO_CHAR(C.PRECISION),null,null, TO_CHAR(C.PRECISION) || DECODE(TO_CHAR(C.SCALE), null,'',',' || TO_CHAR(C.SCALE)) ),  '******ERROR') ||
                   DECODE(C.COLUMN_TYPE,'INT',null,'DATE',null,'TIMESTAMP',null,'TIMESTAMP WITH TIME ZONE',null,'INTERVAL DAY TO SECOND',null,'INTERVAL YEAR TO MONTH',null,
                               'LONG',null,'CLOB',null,'NCLOB',null,'SYS.ANYDATA',null,'SMALLINT',null,
                               'BLOB',null,'DECIMAL',null,'BINARY_DOUBLE',null,'LONG RAW',null,
                               'NUMBER',DECODE(TO_CHAR(C.PRECISION),null,null,')'),')')
                   ||',') ORDER BY C.COLUMN_ORDER ).EXTRACT('/E/text()').GETCLOBVAL(), ',')) ||'''' ||', '
                 || CHR(10) 
                 || '  format => ' || file_format
                 || CHR(10) 
                 || ');' externalTableDdl
      FROM
          MD_COLUMNS C
      WHERE
        C.TYPE = 'CONVERTED'
        AND C.PROJECT_ID = projectID
        AND C.CONNECTION_ID =  connectionID
        AND C.CATALOG_NAME = catalogName
        AND C.SCHEMA_NAME = schemaName
        AND C.TABLE_NAME = tableName
        AND C.CONV_VERSION = g_convertVersionId 
        GROUP BY C.TABLE_NAME, C.SCHEMA_NAME
      ) DUAL ;

    -- DBMS_OUTPUT.PUT_LINE ('output table: ' || clob_result);
    return clob_result;

  END generate_OCAExternalTable;
  
FUNCTION getDTtype(
	   projectID MD_PROJECTS.PROJECT_ID%TYPE, 
	   connectionID MD_CONNECTIONS.CONNECTION_ID%TYPE, 
	   catalogName MD_CATALOGS.CATALOG_NAME%TYPE, 
	   schemaName MD_SCHEMAS.SCHEMA_NAME%TYPE, 
	   tableName MD_TABLES.TABLE_NAME%TYPE 
	   ) RETURN VARCHAR2 IS
v_coltype VARCHAR2(128) := NULL;
BEGIN
	SELECT
       COLUMN_TYPE INTO v_coltype 
    FROM
       MD_COLUMNS 
    WHERE
       TYPE = 'CONVERTED' 
       AND PROJECT_ID = projectID
       AND CONNECTION_ID = connectionID
       AND CATALOG_NAME = catalogName
       AND SCHEMA_NAME = schemaName
       AND TABLE_NAME = tableName
       AND 
       (
          COLUMN_TYPE = 'DATE' 
          OR COLUMN_TYPE = 'TIMESTAMP' 
          OR COLUMN_TYPE = 'TIMESTAMP WITH TIME ZONE' 
          OR COLUMN_TYPE = 'TIMESTAMP WITH LOCAL TIME ZONE' 
       ) FETCH FIRST 1 ROW ONLY;
    
    IF v_coltype IS NOT NULL THEN
      RETURN v_coltype;
    ELSE
      RETURN NULL;
    END IF;
EXCEPTION    
  WHEN OTHERS THEN
    DBMS_OUTPUT.PUT_LINE(SQLERRM);
    DBMS_OUTPUT.PUT_LINE(DBMS_UTILITY.FORMAT_ERROR_STACK());
    RETURN NULL;
END getDTtype;

FUNCTION generate_DTFormats(dttype VARCHAR2) RETURN CLOB IS
l_dtformats CLOB;
BEGIN
  IF UPPER(dttype) = 'DATE' THEN
    l_dtformats := q'[
     DT_FMTS := VARCHAR2_ARRAY(
       'auto',
       'fxyyyy-mm-dd hh24:mi:ss',
       'fxyyyymmdd hh24:mi:ss',
       'fxfmyyyy.mm.dd',                  --ANSI(with Century)
       'fxfmyyyy-mm-dd',                  --ANSI(with Century) 
	   'fxfmyyyy/mm/dd',                  --Japan(with Century)
       'fxfmyy/mm/dd',                    --Japan(without Century)
       'fxfmyy.mm.dd',                    --ANSI(without Century)
       'fxfmmm/dd/yy',                    --US(without Century)
       'fxfmmm-dd-yy',
       'fxfmmm.dd.yy', 
       'fxfmmm/dd/yyyy',                  --US(with Century)
       'fxfmmm-dd-yyyy',
       'fxfmmm.dd.yyyy',
       'fxfmMon dd, yy',                  --(without Century)
       'fxfmMon dd, yyyy'                 --(with Century)
	   );
    ]';
  ELSIF UPPER(dttype) = 'TIMESTAMP' THEN
    l_dtformats := q'[
     DT_FMTS := VARCHAR2_ARRAY(
       'auto',
	   'fxyyyy-mm-dd hh24:mi:ss',         --ODBC canonical
	   'fxyyyy-mm-dd hh24:mi:ssxff3',     --ODBC canonical (with milliseconds)
	   'fxyyyy-mm-dd"T"hh12:mi:ssxff3',   --ISO8601 (no spaces)
	   'fxyyyy-mm-dd hh12:mi:ssxff3'      --ISO8601 (with space)
	   );
    ]';  
  ELSIF UPPER(dttype) = 'TIMESTAMP WITH TIME ZONE' THEN
    l_dtformats := q'[
     DT_FMTS := VARCHAR2_ARRAY(
       'auto',
       'fxyyyy-mm-dd hh24:mi:ss+tzr',
       'dd mon yyyy hh24:mi:ss.ff tzr',
	   'mm/dd/yyyy hh24:mi:ss.ff tzr',
	   '	yyyy-mm-dd hh24:mi:ss+/-tzr',
	   'yyyy-mm-dd hh24:mi:ss.ff3',
	   '	dd.mm.yyyy hh24:mi:ss tzr',  
       'fxyyyy-mm-dd hh24:mi:ssxff3tzr'  --ISO8601 with time zone Z
	   );
    ]';
  ELSIF UPPER(dttype) = 'TIMESTAMP WITH LOCAL TIME ZONE' THEN
    l_dtformats := q'[
     DT_FMTS := VARCHAR2_ARRAY(
       'auto',
       'dd mon yyyy hh24:mi:ss.ff tzr',
	   'mm/dd/yyyy hh24:mi:ss.ff tzr',
	   'yyyy-mm-dd hh24:mi:ss+/-tzr',
	   'yyyy-mm-dd hh24:mi:ss.ff3',
       'yyyy-mm-dd hh24:mi:ss.ff3tzr',
	   'dd.mm.yyyy hh24:mi:ss tzr'
	   );
    ]';
  END IF;
  RETURN l_dtformats;
END generate_DTFormats;

FUNCTION generate_DTCopyData(
   schemaname MD_SCHEMAS.ORACLE_NAME%TYPE, 
   tablename MD_TABLES.ORACLE_NAME%TYPE,
   credential VARCHAR2,
   fileuri VARCHAR2,
   filefmt VARCHAR2,
   dttype VARCHAR2
   ) RETURN CLOB  IS
 l_dtTpl CLOB;
BEGIN
   l_dtTpl := q'[
DECLARE
	 TYPE VARCHAR2_ARRAY IS TABLE OF VARCHAR2(1000);
	 DT_FMTS VARCHAR2_ARRAY;
	 PROCEDURE COPY_DATA(TABLE_NAME IN VARCHAR2, CREDENTIAL_NAME IN VARCHAR2, FILE_URI_LIST IN CLOB, SCHEMA_NAME IN VARCHAR2,          
	                FORMAT IN CLOB) AS
	 l_opid NUMBER := 0;
	 l_logtable VARCHAR2(200);
	 l_status VARCHAR2(100);
	 l_cursor SYS_REFCURSOR;
	 l_stmt VARCHAR2(4000);
	 l_msg VARCHAR2(4000);
	 l_json JSON_OBJECT_T;
	 BEGIN
	    FOR i in DT_FMTS.FIRST .. DT_FMTS.LAST
	    LOOP
	      BEGIN
	        l_json := JSON_OBJECT_T.parse(FORMAT);
	        l_json.put('<DTFORMAT>', DT_FMTS(i));
            IF l_json.to_string IS JSON THEN 
                DBMS_OUTPUT.PUT_LINE(l_json.to_string);
                DBMS_CLOUD.COPY_DATA (
                  schema_name => SCHEMA_NAME, 
                  table_name => TABLE_NAME, 
                  credential_name => CREDENTIAL_NAME, 
                  file_uri_list => FILE_URI_LIST,
                  format => l_json.to_string,
                  operation_id => l_opid
                );
           END IF;     
           EXIT;
        EXCEPTION
           WHEN OTHERS THEN
            -- DBMS_OUTPUT.PUT_LINE(DBMS_UTILITY.FORMAT_ERROR_STACK());
            IF l_opid > 0 THEN
              l_stmt := 'SELECT STATUS, LOGFILE_TABLE FROM USER_LOAD_OPERATIONS WHERE ID = ' || l_opid ;
              OPEN l_cursor FOR l_stmt;
                 LOOP
                   FETCH l_cursor INTO l_status,l_logtable;
                   EXIT WHEN l_cursor%notfound;
                 END LOOP;
              CLOSE l_cursor;
               
              IF l_status = 'FAILED' THEN
                l_stmt := 'SELECT COUNT(*) FROM ' || l_logtable || ' WHERE REGEXP_INSTR(RECORD,''ORA-26041:|ORA-018\d{2}:'') > 0';
                OPEN l_cursor FOR l_stmt;
                    LOOP
                      FETCH l_cursor INTO l_msg;
                      EXIT WHEN l_cursor%notfound;
                    END LOOP;
                CLOSE l_cursor;
                  
                IF l_msg IS NOT NULL AND TO_NUMBER(l_msg) > 0 THEN
                  CONTINUE;
                ELSE
                  EXIT;
                END IF;
              ELSE
                EXIT;
              END IF;
            ELSE
              EXIT;
            END IF;
        END;
	    END LOOP; 
	 END;
BEGIN
]';
  l_dtTpl :=  l_dtTpl || generate_DTFormats(dttype);
	l_dtTpl := l_dtTpl || q'[
	 COPY_DATA(
      SCHEMA_NAME => '<SCHEMANAME>',
      TABLE_NAME => '<TABLENAME>', 
      CREDENTIAL_NAME => '<CREDENTIALNAME>', 
      FILE_URI_LIST => '<FILEURI>',
      FORMAT => <JSONSTR>
   );
END;
]';

   l_dtTpl := REPLACE(l_dtTpl, '<SCHEMANAME>',  schemaname);
   l_dtTpl := REPLACE(l_dtTpl, '<TABLENAME>',  tablename);
   l_dtTpl := REPLACE(l_dtTpl, '<CREDENTIALNAME>', credential);
   l_dtTpl := REPLACE(l_dtTpl, '<FILEURI>', fileuri);
   l_dtTpl := REPLACE(l_dtTpl, '<JSONSTR>', filefmt);
   IF UPPER(dttype) = 'DATE' THEN
      l_dtTpl := REPLACE(l_dtTpl, '<DTFORMAT>', 'dateformat');
   ELSIF UPPER(dttype) = 'TIMESTAMP' THEN
      l_dtTpl := REPLACE(l_dtTpl, '<DTFORMAT>', 'timestampformat');
   ELSIF UPPER(dttype) = 'TIMESTAMP WITH TIME ZONE' THEN
      l_dtTpl := REPLACE(l_dtTpl, '<DTFORMAT>', 'timestamptzformat');
   ELSIF UPPER(dttype) = 'TIMESTAMP WITH LOCAL TIME ZONE' THEN
      l_dtTpl := REPLACE(l_dtTpl, '<DTFORMAT>', 'timestampltzformat');
   END IF; 
   
   RETURN l_dtTpl;

END generate_DTCopyData;

FUNCTION generate_DTCopyData(
   projectID MD_PROJECTS.PROJECT_ID%TYPE, 
   connectionID MD_CONNECTIONS.CONNECTION_ID%TYPE, 
   catalogName MD_CATALOGS.CATALOG_NAME%TYPE, 
   schemaName MD_SCHEMAS.SCHEMA_NAME%TYPE, 
   schemaOracleName MD_SCHEMAS.ORACLE_NAME%TYPE,
   tableName MD_TABLES.TABLE_NAME%TYPE, 
   source_table_num_rows MD_TABLES.NUM_ROWS%TYPE, 
   tableOracleName MD_TABLES.ORACLE_NAME%TYPE,
   captureVersion MD_GENERATE.CAPT_VERSION%TYPE, 
   convertVersion MD_GENERATE.CONV_VERSION%TYPE, 
   generateVersion MD_GENERATE.GENERATE_VERSION%TYPE,
   credential_name VARCHAR2,
   file_uri_list VARCHAR2,
   file_format VARCHAR2
   ) RETURN CLOB  IS
 l_dtTpl CLOB;
 l_stmt CLOB;
 l_stmt1 CLOB;
 l_dttypescur SYS_REFCURSOR;
 l_dttype VARCHAR2(128);
 l_datefmts CLOB;
 l_tsfmts CLOB;
 l_tstzfmts CLOB;
 l_tstlzfmts CLOB;

 
BEGIN
   l_dtTpl := q'[
DECLARE
	TYPE VARCHAR2_ARRAY IS TABLE OF VARCHAR2(1000);
	DT_FMTS VARCHAR2_ARRAY;
	DT_DTFMTS VARCHAR2_ARRAY;
	DT_TSFMTS VARCHAR2_ARRAY;
	DT_TSTZTS VARCHAR2_ARRAY;
	DT_TSLTZTS VARCHAR2_ARRAY;
	TYPE TBLDTTypes IS TABLE OF VARCHAR2(128) INDEX BY VARCHAR2(1000);
	Tbl_to_DTTyp TBLDTTypes;
	l_curid NUMBER;
	l_desctab    DBMS_SQL.DESC_TAB3;
	l_colcnt     NUMBER;
	l_tgttbl_stmt   VARCHAR2(2000);
   
	PROCEDURE COPY_DATA(PROJECT_ID IN NUMBER, CONNECTION_ID IN NUMBER, CATALOG_NAME IN VARCHAR2, SOURCE_SCHEMA_NAME IN VARCHAR2, SOURCE_TABLE_NAME IN VARCHAR2, SOURCE_TABLE_ROWS IN NUMBER, TARGET_TABLE_NAME IN VARCHAR2, CAPT_VERSION IN NUMBER, CONV_VERSION IN NUMBER, GENERATE_VERSION IN NUMBER, CREDENTIAL_NAME IN VARCHAR2, FILE_URI_LIST IN CLOB, 
	                    TARGET_SCHEMA_NAME IN VARCHAR2, FORMAT IN CLOB) AS
		l_opid NUMBER := 0;
		l_logtable VARCHAR2(200);
		l_status VARCHAR2(100);
		l_cursor SYS_REFCURSOR;
		l_dttypescur SYS_REFCURSOR;
		l_stmt VARCHAR2(4000);
		l_msg VARCHAR2(4000);
		l_json JSON_OBJECT_T;
		l_datamoved BOOLEAN := FALSE;
		l_dtfmtsIdx NUMBER := 1;
		l_tsfmtsIdx NUMBER := 1;
		l_tstzfmtsIdx NUMBER := 1;
		l_tsltzfmtsIdx NUMBER := 1;
		l_dttype VARCHAR2(200) := NULL;
		l_dtcolumn VARCHAR2(200) := NULL;
		l_curid NUMBER;
		l_desctab    DBMS_SQL.DESC_TAB3;
		l_colcnt     NUMBER;
		l_tgttbl_stmt   VARCHAR2(2000);
		error_message varchar2(2000);
		source_num_rows NUMBER;
		owner_name1 VARCHAR2(128);
		rows_loaded1 NUMBER;
		status1 VARCHAR2(9);
		logfile_table1 VARCHAR2(128);
		badfile_table1 VARCHAR2(128);
		start_time1 TIMESTAMP(6) WITH TIME ZONE;
		update_time1 TIMESTAMP(6) WITH TIME ZONE;
		
	 BEGIN
        l_json := JSON_OBJECT_T.parse(FORMAT);
        l_dtcolumn := Tbl_to_DTTyp.FIRST;
        WHILE l_dtcolumn IS NOT NULL LOOP
            IF Tbl_to_DTTyp(l_dtcolumn) = 'DATE' THEN
                l_json.put('dateformat',DT_DTFMTS(l_dtfmtsIdx) );
            ELSIF Tbl_to_DTTyp(l_dtcolumn) = 'TIMESTAMP' THEN
                l_json.put('timestampformat',DT_TSFMTS(l_tsfmtsIdx) );
            ELSIF Tbl_to_DTTyp(l_dtcolumn) = 'TIMESTAMP WITH TIME ZONE' THEN
                l_json.put('timestamptzformat',DT_TSTZTS(l_tstzfmtsIdx) );
            ELSIF Tbl_to_DTTyp(l_dtcolumn) = 'TIMESTAMP WITH LOCAL TIME ZONE' THEN
                l_json.put('timestampltzformat',DT_TSLTZTS(l_tsltzfmtsIdx) );
            END IF;
            l_dtcolumn  := tbl_to_DTTyp.NEXT(l_dtcolumn);
        END LOOP;  

		-- DBMS_OUTPUT.PUT_LINE('JSON: ' || l_json.to_string);
        WHILE NOT l_datamoved LOOP
			BEGIN
			  IF l_json.to_string IS JSON THEN 
			        -- DBMS_OUTPUT.PUT_LINE(l_json.to_string);
			        DBMS_CLOUD.COPY_DATA (
			          schema_name => TARGET_SCHEMA_NAME, 
			          table_name => TARGET_TABLE_NAME, 
			          credential_name => CREDENTIAL_NAME, 
			          file_uri_list => FILE_URI_LIST,
			          format => l_json.to_string,
			          operation_id => l_opid
			        );
			  END IF;     
			  l_datamoved := TRUE;
			EXCEPTION
			  WHEN OTHERS THEN
			    -- DBMS_OUTPUT.PUT_LINE(DBMS_UTILITY.FORMAT_ERROR_STACK());
			    IF l_opid > 0 THEN
			      l_stmt := 'SELECT STATUS, LOGFILE_TABLE FROM USER_LOAD_OPERATIONS WHERE ID = ' || l_opid ;
			      OPEN l_cursor FOR l_stmt;
			         LOOP
			           FETCH l_cursor INTO l_status,l_logtable;
			           EXIT WHEN l_cursor%notfound;
			         END LOOP;
			      CLOSE l_cursor;
			
			      IF l_status = 'FAILED' THEN
			        -- DBMS_OUTPUT.PUT_LINE('FAILED SO TRY DIFFERENT FORMATS HERE');
			        l_stmt := 'SELECT COUNT(*) FROM ' || l_logtable || ' WHERE REGEXP_INSTR(RECORD,''ORA-26041:|ORA-018\d{2}:'') > 0';
			        OPEN l_cursor FOR l_stmt;
			            LOOP
			              FETCH l_cursor INTO l_msg;
			              EXIT WHEN l_cursor%notfound;
			            END LOOP;
			        CLOSE l_cursor;
			
			        IF l_msg IS NOT NULL AND TO_NUMBER(l_msg) > 0 THEN
			            l_stmt := 'SELECT REGEXP_SUBSTR(RECORD, ''error processing column\s(\w+).*'', 1, 1, ''i'', 1) COL FROM <LOGTABLE>
			                               WHERE RECORD LIKE ''%error processing column%''';
			            l_stmt := REPLACE(l_stmt, '<LOGTABLE>', l_logtable); 
			                                  
			           OPEN l_cursor FOR l_stmt;
			            LOOP
			              FETCH l_cursor INTO l_dtcolumn;
			              EXIT WHEN l_cursor%notfound;
			            END LOOP;
			           CLOSE l_cursor;
			           
			           -- DBMS_OUTPUT.PUT_LINE('FAILED DATE TYPE IS ' || l_dttype);
			           IF l_dtcolumn IS NULL OR LENGTH(l_dtcolumn) = 0 THEN
			             EXIT;
			           ELSE
			            IF Tbl_to_DTTyp(l_dtcolumn) = 'DATE' THEN
			                IF l_dtfmtsIdx IS NOT NULL THEN
			                    l_dtfmtsIdx := DT_DTFMTS.NEXT(l_dtfmtsIdx);
			                    IF l_dtfmtsIdx IS NULL THEN
			                      EXIT;
			                    END IF;
			                    -- DBMS_OUTPUT.PUT_LINE('TRY DIFFERENT DATE FORMAT');
			                    l_json.put('dateformat', DT_DTFMTS(l_dtfmtsIdx));
			                END IF;
			            ELSIF Tbl_to_DTTyp(l_dtcolumn) = 'TIMESTAMP' THEN
			                IF l_tsfmtsIdx IS NOT NULL THEN
			                    l_tsfmtsIdx := DT_TSFMTS.NEXT(l_tsfmtsIdx);
			                    IF l_tsfmtsIdx IS NULL THEN
			                      EXIT;
			                    END IF;
			                    -- DBMS_OUTPUT.PUT_LINE('TRY DIFFERENT TIMESTAMP FORMAT');
			                    l_json.put('timestampformat', DT_TSFMTS(l_tsfmtsIdx));
			                END IF;
			            ELSIF Tbl_to_DTTyp(l_dtcolumn) = 'TIMESTAMP WITH TIME ZONE' THEN
			                IF l_tstzfmtsIdx IS NOT NULL THEN
			                    l_tstzfmtsIdx := DT_TSTZTS.NEXT(l_tstzfmtsIdx);
			                    IF l_tstzfmtsIdx IS NULL THEN
			                      EXIT;
			                    END IF;
			                    -- DBMS_OUTPUT.PUT_LINE('TRY DIFFERENT TIMESTAMP WITH TIME ZONE FORMAT');
			                    l_json.put('timestamptzformat', DT_TSTZTS(l_tstzfmtsIdx));
			                END IF;
			            ELSIF Tbl_to_DTTyp(l_dtcolumn) = 'TIMESTAMP WITH LOCAL TIME ZONE' THEN
			                IF l_tsltzfmtsIdx IS NOT NULL THEN
			                    l_tsltzfmtsIdx := DT_TSLTZTS.NEXT(l_tsltzfmtsIdx);
			                    IF l_tsltzfmtsIdx IS NULL THEN
			                      EXIT;
			                    END IF;
			                    -- DBMS_OUTPUT.PUT_LINE('TRY DIFFERENT TIMESTAMP WITH LOCAL TIME ZONE FORMAT');
			                    l_json.put('timestampltzformat', DT_TSLTZTS(l_tsltzfmtsIdx));
			                END IF;
			            END IF;
			          END IF;
			          CONTINUE;
			        ELSE
			          EXIT;
			        END IF;
			      ELSE
			        EXIT;
			      END IF;
			    ELSE
			      EXIT;
			    END IF;
			END;
		END LOOP;
				   
           select owner_name, rows_loaded, status, logfile_table, badfile_table, start_time, update_time 
	       into owner_name1, rows_loaded1, status1, logfile_table1, badfile_table1, start_time1, update_time1 
	       from dba_load_operations where ID = l_opid;
	 
	       DBMS_OUTPUT.PUT_LINE(owner_name1||' ' ||rows_loaded1||' ' || status1||' ' || logfile_table1||' ' || badfile_table1||' ' || start_time1||' ' ||update_time1 );
	
	       INSERT into SQLDEV_MIGREPOS.MD_REPORT(PROJECT_ID, CONNECTION_ID, CATALOG_NAME, CAPT_VERSION, CONV_VERSION, GENERATE_VERSION, OPERATION_ID, SOURCE_SCHEMA_NAME, SOURCE_TABLE_NAME, SOURCE_TABLE_ROWS,
	                                      TARGET_SCHEMA_NAME, TARGET_TABLE_NAME, TARGET_TABLE_ROWS_LOADED, LOGFILE_TABLE, BADFILE_TABLE, STATUS, START_TIME, END_TIME, CREATED_BY) 
	                               values(PROJECT_ID, CONNECTION_ID, CATALOG_NAME, CAPT_VERSION, CONV_VERSION, GENERATE_VERSION, l_opid, SOURCE_SCHEMA_NAME, SOURCE_TABLE_NAME, SOURCE_TABLE_ROWS, 
	                                      TARGET_SCHEMA_NAME, TARGET_TABLE_NAME, rows_loaded1, logfile_table1, badfile_table1, status1, start_time1, update_time1, owner_name1 );
	                                                                           
	       commit;
	
	       EXCEPTION
	          WHEN OTHERS THEN
	          DBMS_OUTPUT.PUT_LINE ('Data load failed');
	          error_message := SUBSTR(SQLERRM, 1, 2000);
	          DBMS_OUTPUT.PUT_LINE ('error_message '||error_message);
	    
	          INSERT into SQLDEV_MIGREPOS.MD_REPORT(PROJECT_ID, CONNECTION_ID, CATALOG_NAME, CAPT_VERSION, CONV_VERSION, GENERATE_VERSION, OPERATION_ID, SOURCE_SCHEMA_NAME, SOURCE_TABLE_NAME, SOURCE_TABLE_ROWS,
	                                      TARGET_SCHEMA_NAME, TARGET_TABLE_NAME, LOGFILE_TABLE, BADFILE_TABLE, STATUS, ERROR_MESSAGES, START_TIME, END_TIME, CREATED_BY) 
	                               values(PROJECT_ID, CONNECTION_ID, CATALOG_NAME, CAPT_VERSION, CONV_VERSION, GENERATE_VERSION, -1, SOURCE_SCHEMA_NAME, SOURCE_TABLE_NAME, SOURCE_TABLE_ROWS, 
	                                       TARGET_SCHEMA_NAME, TARGET_TABLE_NAME, logfile_table1, badfile_table1, 'FAILED', error_message, start_time1, update_time1, owner_name1 );
	       commit;
              
	 END;
BEGIN
	l_tgttbl_stmt := <DTTYPESQRY>;
	l_curid := DBMS_SQL.OPEN_CURSOR;
	DBMS_SQL.PARSE(l_curid, l_tgttbl_stmt, DBMS_SQL.NATIVE);
	DBMS_SQL.DESCRIBE_COLUMNS3(l_curid, l_colcnt, l_desctab);
	FOR i IN 1 .. l_colcnt LOOP
		IF l_desctab(i).col_type = 12 THEN
			Tbl_to_DTTyp(l_desctab(i).col_name) := 'DATE';
		ELSIF l_desctab(i).col_type = 180 THEN
			Tbl_to_DTTyp(l_desctab(i).col_name) := 'TIMESTAMP';
		ELSIF l_desctab(i).col_type = 181 THEN
			Tbl_to_DTTyp(l_desctab(i).col_name) := 'TIMESTAMP WITH TIME ZONE';
		ELSIF l_desctab(i).col_type = 231 THEN
			Tbl_to_DTTyp(l_desctab(i).col_name) := 'TIMESTAMP WITH LOCAL TIME ZONE';
		END IF;
	END LOOP;
	DBMS_SQL.CLOSE_CURSOR(l_curid);
]';	
    
   l_datefmts := q'[
	DT_DTFMTS := VARCHAR2_ARRAY(
		'auto',
		'fxyyyy-mm-dd hh24:mi:ss',
		'fxyyyymmdd hh24:mi:ss',
		'fxfmyyyy.mm.dd',                  --ANSI(with Century)
		'fxfmyyyy-mm-dd',                  --ANSI(with Century) 
		'fxfmyyyy/mm/dd',                  --Japan(with Century)
		'fxfmyy/mm/dd',                    --Japan(without Century)
		'fxfmyy.mm.dd',                    --ANSI(without Century)
		'fxfmmm/dd/yy',                    --US(without Century)
		'fxfmmm-dd-yy',
		'fxfmmm.dd.yy', 
		'fxfmmm/dd/yyyy',                  --US(with Century)
		'fxfmmm-dd-yyyy',
		'fxfmmm.dd.yyyy',
		'fxfmMon dd, yy',                  --(without Century)
		'fxfmMon dd, yyyy',                --(with Century)
		'yyyy-mm-dd hh24:mi:ss'
	);   
   ]';
   
   l_tsfmts := q'[
	DT_TSFMTS := VARCHAR2_ARRAY(
		'auto',
		'fxyyyy-mm-dd hh24:mi:ss',         --ODBC canonical
		'fxyyyy-mm-dd hh24:mi:ssxff3',     --ODBC canonical (with milliseconds)
		'fxyyyy-mm-dd"T"hh12:mi:ssxff3',   --ISO8601 (no spaces)
		'fxyyyy-mm-dd hh12:mi:ssxff3',     --ISO8601 (with space)
		'yyyy-mm-dd hh24:mi:ssxff'
	);
   ]';

   l_tstzfmts := q'[
	DT_TSTZTS := VARCHAR2_ARRAY(
		'auto',
		'fxyyyy-mm-dd hh24:mi:ss+tzr',
		'dd mon yyyy hh24:mi:ss.ff tzr',
		'mm/dd/yyyy hh24:mi:ss.ff tzr',
		'yyyy-mm-dd hh24:mi:ss+/-tzr',
		'yyyy-mm-dd hh24:mi:ss.ff3',
		'dd.mm.yyyy hh24:mi:ss tzr',  
		'fxyyyy-mm-dd hh24:mi:ssxff3tzr', --ISO8601 with time zone Z
		'yyyy-mm-dd hh24:mi:ssxfftzh:tzm'
	);
   ]';
   
   l_tstlzfmts := q'[
	DT_TSLTZTS := VARCHAR2_ARRAY(
		'auto',
		'dd mon yyyy hh24:mi:ss.ff tzr',
		'mm/dd/yyyy hh24:mi:ss.ff tzr',
		'yyyy-mm-dd hh24:mi:ss+/-tzr',
		'yyyy-mm-dd hh24:mi:ss.ff3',
		'yyyy-mm-dd hh24:mi:ss.ff3tzr',
		'dd.mm.yyyy hh24:mi:ss tzr'
	);
   ]';
   
   l_stmt1 := q'[SELECT DISTINCT
                    COLUMN_TYPE
                FROM
                    MD_COLUMNS
                WHERE
                    TYPE = 'CONVERTED'
                    AND PROJECT_ID = <PROJECTID>
                    AND CONNECTION_ID = <CONNID>
                    AND CATALOG_NAME = '<CATALOGNAME>'
                    AND SCHEMA_NAME = '<SCHEMANAME>'
                    AND TABLE_NAME = '<TABLENAME>'
                    AND CAPT_VERSION = '<CAPT_VERSION>'
                    AND CONV_VERSION = '<CONV_VERSION>'
                    AND (COLUMN_TYPE = 'DATE'
                          OR COLUMN_TYPE = 'TIMESTAMP'
                          OR COLUMN_TYPE = 'TIMESTAMP WITH TIME ZONE'
                          OR COLUMN_TYPE = 'TIMESTAMP WITH LOCAL TIME ZONE')
                ORDER BY
                    COLUMN_TYPE]';
                    
   l_stmt1 := REPLACE(l_stmt1, '<PROJECTID>',  projectID);
   l_stmt1 := REPLACE(l_stmt1, '<CONNID>',  connectionID);
   l_stmt1 := REPLACE(l_stmt1, '<CATALOGNAME>',  catalogName);
   l_stmt1 := REPLACE(l_stmt1, '<SCHEMANAME>',  schemaName);
   l_stmt1 := REPLACE(l_stmt1, '<TABLENAME>',  tableName);
   l_stmt1 := REPLACE(l_stmt1, '<CAPT_VERSION>',  captureVersion);
   l_stmt1 := REPLACE(l_stmt1, '<CONV_VERSION>',  convertVersion);
   
	OPEN l_dttypescur FOR l_stmt1; 
	LOOP
		FETCH l_dttypescur INTO l_dttype;
		EXIT WHEN l_dttypescur%notfound;
	IF l_dttype = 'DATE' THEN
	l_dtTpl := l_dtTpl || q'[
	<DTFMTS>]';
	ELSIF l_dttype = 'TIMESTAMP' THEN
	l_dtTpl := l_dtTpl || q'[
    <DTTSFMTS>]';
	ELSIF l_dttype = 'TIMESTAMP WITH TIME ZONE' THEN
	l_dtTpl := l_dtTpl || q'[
	<DTTSTZFMTS>]';
	ELSIF l_dttype = 'TIMESTAMP WITH LOCAL TIME ZONE' THEN
	l_dtTpl := l_dtTpl || q'[
	<DTTSTLTZFMTS>]';
	END IF;
	END LOOP;
	CLOSE l_dttypescur;
   
   
   
   l_dtTpl := l_dtTpl || q'[
	COPY_DATA(
		PROJECT_ID => '<PROJECTID>',
   		CONNECTION_ID => '<CONNID>', 
   		CATALOG_NAME => '<CATALOGNAME>',
	    SOURCE_SCHEMA_NAME => '<SCHEMANAME>',
		SOURCE_TABLE_NAME => '<TABLENAME>',
		SOURCE_TABLE_ROWS => '<SOURCE_TABLE_ROWS>',
		TARGET_SCHEMA_NAME => '<SCHEMAORACLENAME>',
		TARGET_TABLE_NAME => '<TABLEORACLENAME>',
		CAPT_VERSION => '<CAPT_VERSION>',
		CONV_VERSION => '<CONV_VERSION>',
		GENERATE_VERSION => '<GENERATE_VERSION>',
		CREDENTIAL_NAME => '<CREDENTIALNAME>', 
		FILE_URI_LIST => '<FILEURI>',
		FORMAT => <JSONSTR>
    );
END;
]';
   
   l_stmt := q'['SELECT * FROM <SCHEMAORACLENAME>.<TABLEORACLENAME>']';
   l_stmt := REPLACE(l_stmt, '<SCHEMAORACLENAME>',  schemaOracleName);
   l_stmt := REPLACE(l_stmt, '<TABLEORACLENAME>',  tableOracleName);
   
   l_dtTpl := REPLACE(l_dtTpl, '<DTTYPESQRY>',  l_stmt);
   
   l_dtTpl := REPLACE(l_dtTpl, '<PROJECTID>', projectID);
   l_dtTpl := REPLACE(l_dtTpl, '<CONNID>', connectionID);
   l_dtTpl := REPLACE(l_dtTpl, '<CATALOGNAME>', catalogName);
   
   l_dtTpl := REPLACE(l_dtTpl, '<SCHEMANAME>',  schemaName);
   l_dtTpl := REPLACE(l_dtTpl, '<TABLENAME>',  tableName);
   l_dtTpl := REPLACE(l_dtTpl, '<SOURCE_TABLE_ROWS>',  source_table_num_rows);
   l_dtTpl := REPLACE(l_dtTpl, '<SCHEMAORACLENAME>',  schemaOracleName);
   l_dtTpl := REPLACE(l_dtTpl, '<TABLEORACLENAME>',  tableOracleName);
   l_dtTpl := REPLACE(l_dtTpl, '<CREDENTIALNAME>', credential_name);
   
   l_dtTpl := REPLACE(l_dtTpl, '<CAPT_VERSION>',  captureVersion);
   l_dtTpl := REPLACE(l_dtTpl, '<CONV_VERSION>',  convertVersion);
   l_dtTpl := REPLACE(l_dtTpl, '<GENERATE_VERSION>',  generateVersion);
   
   l_dtTpl := REPLACE(l_dtTpl, '<FILEURI>', file_uri_list);
   l_dtTpl := REPLACE(l_dtTpl, '<JSONSTR>', file_format);
   
   l_dtTpl := REPLACE(l_dtTpl, '<DTFMTS>', l_datefmts);
   l_dtTpl := REPLACE(l_dtTpl, '<DTTSFMTS>', l_tsfmts);
   l_dtTpl := REPLACE(l_dtTpl, '<DTTSTZFMTS>', l_tstzfmts);
   l_dtTpl := REPLACE(l_dtTpl, '<DTTSTLTZFMTS>', l_tstlzfmts);
   
   RETURN l_dtTpl;

END generate_DTCopyData;

/*
FUNCTION generate_OCACopyData( 
   projectID MD_PROJECTS.PROJECT_ID%TYPE, 
   connectionID MD_CONNECTIONS.CONNECTION_ID%TYPE, 
   catalogName MD_CATALOGS.CATALOG_NAME%TYPE, 
   schemaName MD_SCHEMAS.SCHEMA_NAME%TYPE, 
   schemaOracleName MD_SCHEMAS.ORACLE_NAME%TYPE, 
   tableName MD_TABLES.TABLE_NAME%TYPE, 
   source_table_num_rows MD_TABLES.NUM_ROWS%TYPE,
   tableOracleName MD_TABLES.ORACLE_NAME%TYPE,
   captureVersion MD_GENERATE.CAPT_VERSION%TYPE, 
   convertVersion MD_GENERATE.CONV_VERSION%TYPE, 
   generateVersion MD_GENERATE.GENERATE_VERSION%TYPE,
   credential_name VARCHAR2,
   file_uri_list VARCHAR2,
   file_format VARCHAR2
   ) RETURN CLOB  IS

  clob_result CLOB;
  dtType VARCHAR2(128) := NULL;

  BEGIN
	  dtType := getDTtype(projectID, connectionID, catalogName, schemaName, tableName);
	  DBMS_OUTPUT.PUT_LINE ('dtType: ' || dtType);
	  IF dtType IS NOT NULL THEN
         
        clob_result := generate_DTCopyData(
		   projectID=>projectID, 
		   connectionID=>connectionID, 
		   catalogName=>catalogName, 
		   schemaName=>schemaName, 
		   schemaOracleName=>upper(schemaOracleName), 
		   tableName=>tableName, 
		   source_table_num_rows=>source_table_num_rows,  
		   tableOracleName=>upper(tableOracleName),
		   captureVersion=>captureVersion, 
   		   convertVersion=>convertVersion, 
   		   generateVersion=>generateVersion,
		   credential_name=>credential_name ,
		   file_uri_list=>file_uri_list || '/' || schemaOracleName||'/'|| tableOracleName ||'/'|| '*.gz',
		   file_format=>file_format
		   );
	  ELSE
		 SELECT TO_CLOB(copyDataDdl) into clob_result FROM
		  (
		    SELECT 'DECLARE' || CHR(10) ||
                       'v_op_id NUMBER;' || CHR(10) ||
                       'error_message varchar2(2000);'  || CHR(10) ||
		           'BEGIN' || CHR(10) ||
		           'DBMS_CLOUD.copy_data (' || CHR(10)  
		               || '  schema_name => '''  ||  upper(schemaOracleName) || ''', ' || CHR(10) 
		               || '  table_name => '''  ||  upper(tableOracleName) ||''', ' || CHR(10) 
		               || '  credential_name => '''  ||  credential_name ||''', '|| CHR(10) 
		               || '  file_uri_list => '''  ||  file_uri_list ||'/' || schemaOracleName||'/'|| tableOracleName ||'/'|| '*.gz' ||''', ' || CHR(10)
		               || '  format => ' || file_format ||', '|| CHR(10)
		               || '  OPERATION_ID => v_op_id '
		               || CHR(10)
		               || ');' ||CHR(10)
		               || 'INSERT into SQLDEV_MIGREPOS.MD_REPORT(PROJECT_ID, CONNECTION_ID, CATALOG_NAME, CAPT_VERSION, CONV_VERSION, GENERATE_VERSION, OPERATION_ID, SOURCE_SCHEMA_NAME, SOURCE_TABLE_NAME, TARGET_SCHEMA_NAME, TARGET_TABLE_NAME, STATUS, START_TIME) ' || CHR(10)
		               || '       values('||projectID||', '||connectionID||', '''||catalogName||''', '||captureVersion||', '||convertVersion||', '||generateVersion||', v_op_id, '''||schemaName||''', '''||tableName||''', '''||upper(schemaOracleName)||''', '''||upper(tableOracleName)||''',''COMPLETED'', SYSTIMESTAMP);' || CHR(10)
		               || 'commit;' || CHR(10) || CHR(10) 
                       || '  EXCEPTION' || CHR(10)
                       || '    WHEN OTHERS THEN' || CHR(10)
                       || '    DBMS_OUTPUT.PUT_LINE ('||'''Data load failed for '||schemaOracleName||'.'||tableOracleName||''');' || CHR(10)
                       || '    error_message := SUBSTR(SQLERRM, 1, 2000);' || CHR(10)
		               || '    INSERT into SQLDEV_MIGREPOS.MD_REPORT(PROJECT_ID, CONNECTION_ID, CATALOG_NAME, CAPT_VERSION, CONV_VERSION, GENERATE_VERSION, OPERATION_ID, SOURCE_SCHEMA_NAME, SOURCE_TABLE_NAME, TARGET_SCHEMA_NAME, TARGET_TABLE_NAME, STATUS, ERROR_MESSAGES, START_TIME) ' || CHR(10)
		               || '           values('||projectID||', '||connectionID||', '''||catalogName||''', '||captureVersion||', '||convertVersion||', '||generateVersion||', -1, '''||schemaName||''', '''||tableName||''', '''||upper(schemaOracleName)||''', '''||upper(tableOracleName)||''',''FAILED'', error_message, SYSTIMESTAMP);' || CHR(10)
		               || '  commit;' || CHR(10)		               
		               ||CHR(10) || 
		            'END;' || CHR(10)  copyDataDdl  
              
              FROM
		          MD_COLUMNS C
		      WHERE
		        C.TYPE = 'CONVERTED'
		        AND C.PROJECT_ID = projectID
		        AND C.CONNECTION_ID =  connectionID
		        AND C.CATALOG_NAME = catalogName
		        AND C.SCHEMA_NAME = schemaName
		        AND C.TABLE_NAME = tableName
		        AND C.CONV_VERSION = g_convertVersionId
		        GROUP BY C.TABLE_NAME, C.SCHEMA_NAME
		      ) DUAL ;
	   END IF;      
	 RETURN clob_result;
 EXCEPTION    
  WHEN OTHERS THEN
    DBMS_OUTPUT.PUT_LINE(SQLERRM);
    DBMS_OUTPUT.PUT_LINE(DBMS_UTILITY.FORMAT_ERROR_STACK());
    RETURN NULL;
 END generate_OCACopyData;
 
 */
 
 FUNCTION generate_OCACopyData( 
   projectID MD_PROJECTS.PROJECT_ID%TYPE, 
   connectionID MD_CONNECTIONS.CONNECTION_ID%TYPE, 
   catalogName MD_CATALOGS.CATALOG_NAME%TYPE, 
   schemaName MD_SCHEMAS.SCHEMA_NAME%TYPE, 
   schemaOracleName MD_SCHEMAS.ORACLE_NAME%TYPE, 
   tableName MD_TABLES.TABLE_NAME%TYPE, 
   tableOracleName MD_TABLES.ORACLE_NAME%TYPE,
   source_table_num_rows MD_TABLES.NUM_ROWS%TYPE,
   captureVersion MD_GENERATE.CAPT_VERSION%TYPE, 
   convertVersion MD_GENERATE.CONV_VERSION%TYPE, 
   generateVersion MD_GENERATE.GENERATE_VERSION%TYPE,
   credential_name VARCHAR2,
   file_uri_list VARCHAR2,
   file_format VARCHAR2
   ) RETURN CLOB  IS

  clob_result CLOB;
  dtType VARCHAR2(128) := NULL;
  clob_caller CLOB;

  BEGIN
	  dtType := getDTtype(projectID, connectionID, catalogName, schemaName, tableName);
	  DBMS_OUTPUT.PUT_LINE ('dtType: ' || dtType);
	  IF dtType IS NOT NULL THEN
                   
        clob_result := generate_DTCopyData(
		   projectID=>projectID, 
		   connectionID=>connectionID, 
		   catalogName=>catalogName, 
		   schemaName=>schemaName, 
		   schemaOracleName=>upper(schemaOracleName), 
		   tableName=>tableName,
		   source_table_num_rows=>source_table_num_rows,  
		   tableOracleName=>upper(tableOracleName),
		   captureVersion=>captureVersion, 
   		   convertVersion=>convertVersion, 
   		   generateVersion=>generateVersion,
		   credential_name=>credential_name ,
		   file_uri_list=>file_uri_list || '/' || schemaOracleName||'/'|| tableOracleName ||'/'|| '*.gz',
		   file_format=>file_format
		   );
	  ELSE
	  
	  clob_caller := q'[ 
DECLARE
l_opid NUMBER;
error_message varchar2(2000);
source_num_rows NUMBER;
owner_name1 VARCHAR2(128);
rows_loaded1 NUMBER;
status1 VARCHAR2(9);
logfile_table1 VARCHAR2(128);
badfile_table1 VARCHAR2(128);
start_time1 TIMESTAMP(6) WITH TIME ZONE;
update_time1 TIMESTAMP(6) WITH TIME ZONE;

BEGIN

	DBMS_CLOUD.copy_data (
	  schema_name => '<TARGET_SCHEMA_NAME>', 
	  table_name => '<TARGET_TABLE_NAME>', 
	  credential_name => '<CREDENTIAL_NAME>', 
	  file_uri_list => '<FILE_URI_LIST>', 
	  format => <FILE_FORMAT>, 
	  OPERATION_ID => l_opid 
	);
	DBMS_OUTPUT.PUT_LINE ('Data load successful operation_id = '||l_opid);
	
	select owner_name, rows_loaded, status, logfile_table, badfile_table, start_time, update_time 
	       into owner_name1, rows_loaded1, status1, logfile_table1, badfile_table1, start_time1, update_time1 
	       from dba_load_operations where ID = l_opid;
	 
	DBMS_OUTPUT.PUT_LINE(owner_name1||' ' ||rows_loaded1||' ' || status1||' ' || logfile_table1||' ' || badfile_table1||' ' || start_time1||' ' ||update_time1 );
	
	INSERT into SQLDEV_MIGREPOS.MD_REPORT(PROJECT_ID, CONNECTION_ID, CATALOG_NAME, CAPT_VERSION, CONV_VERSION, GENERATE_VERSION, OPERATION_ID, SOURCE_SCHEMA_NAME, SOURCE_TABLE_NAME, SOURCE_TABLE_ROWS,
	                                      TARGET_SCHEMA_NAME, TARGET_TABLE_NAME, TARGET_TABLE_ROWS_LOADED, LOGFILE_TABLE, BADFILE_TABLE, STATUS, START_TIME, END_TIME, CREATED_BY) 
	                               values('<PROJECT_ID>', '<CONNECTION_ID>', '<CATALOG_NAME>', '<CAPT_VERSION>', '<CONV_VERSION>', '<GENERATE_VERSION>', l_opid, '<SOURCE_SCHEMA_NAME>', '<SOURCE_TABLE_NAME>', '<SOURCE_TABLE_ROWS>', 
	                                      '<TARGET_SCHEMA_NAME>', '<TARGET_TABLE_NAME>', rows_loaded1, logfile_table1, badfile_table1, status1, start_time1, update_time1, owner_name1 );                                                                       
	commit;
	
	  EXCEPTION
	    WHEN OTHERS THEN
	    DBMS_OUTPUT.PUT_LINE ('Data load failed');
	    error_message := SUBSTR(SQLERRM, 1, 2000);
	    DBMS_OUTPUT.PUT_LINE ('error_message '||error_message);
	    
	    INSERT into SQLDEV_MIGREPOS.MD_REPORT(PROJECT_ID, CONNECTION_ID, CATALOG_NAME, CAPT_VERSION, CONV_VERSION, GENERATE_VERSION, OPERATION_ID, SOURCE_SCHEMA_NAME, SOURCE_TABLE_NAME, SOURCE_TABLE_ROWS,
	                                      TARGET_SCHEMA_NAME, TARGET_TABLE_NAME, LOGFILE_TABLE, BADFILE_TABLE, STATUS, ERROR_MESSAGES, START_TIME, END_TIME, CREATED_BY) 
	                               values('<PROJECT_ID>', '<CONNECTION_ID>', '<CATALOG_NAME>', '<CAPT_VERSION>', '<CONV_VERSION>', '<GENERATE_VERSION>', -1, '<SOURCE_SCHEMA_NAME>', '<SOURCE_TABLE_NAME>', '<SOURCE_TABLE_ROWS>', 
	                                       '<TARGET_SCHEMA_NAME>', '<TARGET_TABLE_NAME>', logfile_table1, badfile_table1, 'FAILED', error_message, start_time1, update_time1, owner_name1 );
	 commit;

END ;
]';
	  
	  	clob_caller := REPLACE(clob_caller, '<PROJECT_ID>',  projectID);
	    clob_caller := REPLACE(clob_caller, '<CONNECTION_ID>',  connectionID);
	    clob_caller := REPLACE(clob_caller, '<CATALOG_NAME>',  catalogName);
	    clob_caller := REPLACE(clob_caller, '<SOURCE_SCHEMA_NAME>',  schemaName);
	    clob_caller := REPLACE(clob_caller, '<SOURCE_TABLE_NAME>',  tableName);
	    clob_caller := REPLACE(clob_caller, '<SOURCE_TABLE_ROWS>', source_table_num_rows );
	    clob_caller := REPLACE(clob_caller, '<TARGET_SCHEMA_NAME>',  upper(schemaOracleName));
	    clob_caller := REPLACE(clob_caller, '<TARGET_TABLE_NAME>',  upper(tableOracleName));
	    clob_caller := REPLACE(clob_caller, '<CAPT_VERSION>',  captureVersion);
	    clob_caller := REPLACE(clob_caller, '<CONV_VERSION>',  convertVersion);
	    clob_caller := REPLACE(clob_caller, '<GENERATE_VERSION>',  generateVersion);
	    clob_caller := REPLACE(clob_caller, '<CREDENTIAL_NAME>',  credential_name);
	    clob_caller := REPLACE(clob_caller, '<FILE_URI_LIST>',  file_uri_list || '/' || schemaOracleName||'/'|| tableOracleName ||'/'|| '*.gz');
	    clob_caller := REPLACE(clob_caller, '<FILE_FORMAT>',  file_format);
		 
		clob_result := clob_caller;
	    
	   END IF;      
	 RETURN clob_result;
 EXCEPTION    
  WHEN OTHERS THEN
    DBMS_OUTPUT.PUT_LINE(SQLERRM);
    DBMS_OUTPUT.PUT_LINE(DBMS_UTILITY.FORMAT_ERROR_STACK());
    RETURN NULL;
 END generate_OCACopyData;
 
 
 FUNCTION generate_OCACopyData_utilPkg( 
   projectID MD_PROJECTS.PROJECT_ID%TYPE, 
   connectionID MD_CONNECTIONS.CONNECTION_ID%TYPE, 
   catalogName MD_CATALOGS.CATALOG_NAME%TYPE, 
   schemaName MD_SCHEMAS.SCHEMA_NAME%TYPE, 
   schemaOracleName MD_SCHEMAS.ORACLE_NAME%TYPE, 
   tableName MD_TABLES.TABLE_NAME%TYPE, 
   tableOracleName MD_TABLES.ORACLE_NAME%TYPE,
   source_table_num_rows MD_TABLES.NUM_ROWS%TYPE,
   captureVersion MD_GENERATE.CAPT_VERSION%TYPE, 
   convertVersion MD_GENERATE.CONV_VERSION%TYPE, 
   generateVersion MD_GENERATE.GENERATE_VERSION%TYPE,
   credential_name VARCHAR2,
   file_uri_list VARCHAR2,
   file_format VARCHAR2
   ) RETURN CLOB  IS

  clob_result CLOB;
  dtType VARCHAR2(128) := NULL;
  clob_caller CLOB;

  BEGIN
	  dtType := getDTtype(projectID, connectionID, catalogName, schemaName, tableName);
	  DBMS_OUTPUT.PUT_LINE ('dtType: ' || dtType);
	  IF dtType IS NOT NULL THEN
/*	  
	  clob_result := generate_DTCopyData(
		   projectID=>projectID, 
		   connectionID=>connectionID, 
		   catalogName=>catalogName, 
		   schemaName=>schemaName, 
		   schemaOracleName=>upper(schemaOracleName), 
		   tableName=>tableName,
		   source_table_num_rows=>source_table_num_rows,  
		   tableOracleName=>upper(tableOracleName),
		   captureVersion=>captureVersion, 
   		   convertVersion=>convertVersion, 
   		   generateVersion=>generateVersion,
		   credential_name=>credential_name ,
		   file_uri_list=>file_uri_list || '/' || schemaOracleName||'/'|| tableOracleName ||'/'|| '*.gz',
		   file_format=>file_format
		   );
*/	  
         clob_caller := q'[    
BEGIN
     REDSHIFT_MIGRATION_UTILITIES.run_data_load_datetimestamp (
       PROJECT_ID => '<PROJECT_ID>',
       CONNECTION_ID => '<CONNECTION_ID>', 
       CATALOG_NAME => '<CATALOG_NAME>',
       SOURCE_SCHEMA_NAME => '<SOURCE_SCHEMA_NAME>',
       SOURCE_TABLE_NAME => '<SOURCE_TABLE_NAME>',
       SOURCE_TABLE_ROWS => '<SOURCE_TABLE_ROWS>',
       TARGET_SCHEMA_NAME => '<TARGET_SCHEMA_NAME>',
       TARGET_TABLE_NAME => '<TARGET_TABLE_NAME>',
       CAPT_VERSION => '<CAPT_VERSION>',
       CONV_VERSION => '<CONV_VERSION>',
       GENERATE_VERSION => '<GENERATE_VERSION>',
       CREDENTIAL_NAME => '<CREDENTIAL_NAME>',
       FILE_URI_LIST => '<FILE_URI_LIST>',
       FILE_FORMAT => <FILE_FORMAT>
     );
END ;
]';

	  ELSE
		 clob_caller := q'[    
BEGIN
     REDSHIFT_MIGRATION_UTILITIES.run_data_load (
       PROJECT_ID => '<PROJECT_ID>',
       CONNECTION_ID => '<CONNECTION_ID>', 
       CATALOG_NAME => '<CATALOG_NAME>',
       SOURCE_SCHEMA_NAME => '<SOURCE_SCHEMA_NAME>',
       SOURCE_TABLE_NAME => '<SOURCE_TABLE_NAME>',
       SOURCE_TABLE_ROWS => '<SOURCE_TABLE_ROWS>',
       TARGET_SCHEMA_NAME => '<TARGET_SCHEMA_NAME>',
       TARGET_TABLE_NAME => '<TARGET_TABLE_NAME>',
       CAPT_VERSION => '<CAPT_VERSION>',
       CONV_VERSION => '<CONV_VERSION>',
       GENERATE_VERSION => '<GENERATE_VERSION>',
       CREDENTIAL_NAME => '<CREDENTIAL_NAME>',
       FILE_URI_LIST => '<FILE_URI_LIST>',
       FILE_FORMAT => <FILE_FORMAT>
     );
END ;
]';
	  
	 END IF; 
		clob_caller := REPLACE(clob_caller, '<PROJECT_ID>',  projectID);
	    clob_caller := REPLACE(clob_caller, '<CONNECTION_ID>',  connectionID);
	    clob_caller := REPLACE(clob_caller, '<CATALOG_NAME>',  catalogName);
	    clob_caller := REPLACE(clob_caller, '<SOURCE_SCHEMA_NAME>',  schemaName);
	    clob_caller := REPLACE(clob_caller, '<SOURCE_TABLE_NAME>',  tableName);
	    clob_caller := REPLACE(clob_caller, '<SOURCE_TABLE_ROWS>', source_table_num_rows );
	    clob_caller := REPLACE(clob_caller, '<TARGET_SCHEMA_NAME>',  upper(schemaOracleName));
	    clob_caller := REPLACE(clob_caller, '<TARGET_TABLE_NAME>',  upper(tableOracleName));
	    clob_caller := REPLACE(clob_caller, '<CAPT_VERSION>',  captureVersion);
	    clob_caller := REPLACE(clob_caller, '<CONV_VERSION>',  convertVersion);
	    clob_caller := REPLACE(clob_caller, '<GENERATE_VERSION>',  generateVersion);
	    clob_caller := REPLACE(clob_caller, '<CREDENTIAL_NAME>',  credential_name);
	    clob_caller := REPLACE(clob_caller, '<FILE_URI_LIST>',  file_uri_list || '/' || schemaOracleName||'/'|| tableOracleName ||'/'|| '*.gz');
	    clob_caller := REPLACE(clob_caller, '<FILE_FORMAT>',  file_format);
		 
		clob_result := clob_caller;
		
     --END IF;    
		
	 RETURN clob_result;
	 
 EXCEPTION    
  WHEN OTHERS THEN
    DBMS_OUTPUT.PUT_LINE(SQLERRM);
    DBMS_OUTPUT.PUT_LINE(DBMS_UTILITY.FORMAT_ERROR_STACK());
    RETURN NULL;
 END generate_OCACopyData_utilPkg;

-- Generate a Job scheduler
  PROCEDURE GenerateSchedulerJobs(starting_date IN VARCHAR2) IS
  CURSOR table_cursor IS 
      select * from md_generate where object_type = 'OCA_COPY_DATA';
      clob_result CLOB;
      start_date varchar2(100);
  BEGIN
    IF starting_date IS NULL then
       start_date := q'[SYSTIMESTAMP]';
    ELSE 
       start_date := starting_date;
    END IF; 
  
	FOR table_rec IN table_cursor LOOP
	
clob_result := q'[ 
-- Run following SQL Script to schedule data load job for <SCHEMA_NAME>.<OBJECT_NAME>
DECLARE
sqlScript varchar2(10000);
BEGIN
       sqlScript := q'{
<SQL_SCRIPT>
       }';
      
     REDSHIFT_MIGRATION_UTILITIES.schedule_dataload_job_and_run (
       PROJECT_ID => '<PROJECT_ID>',
       CONNECTION_ID => '<CONNECTION_ID>', 
       CATALOG_NAME => '<CATALOG_NAME>',
       SOURCE_SCHEMA_NAME => '<SOURCE_SCHEMA_NAME>',
       SOURCE_TABLE_NAME => '<SOURCE_TABLE_NAME>',
       SCHEMA_NAME => '<SCHEMA_NAME>',
       TABLE_NAME => '<OBJECT_NAME>',
       CAPT_VERSION => '<CAPT_VERSION>',
       CONV_VERSION => '<CONV_VERSION>',
       GENERATE_VERSION => '<GENERATE_VERSION>',
       JOBACTION => sqlScript, 
       STARTDATE => '<START_DATE>'
     );
END ;
]';
	    clob_result := REPLACE(clob_result, '<PROJECT_ID>',  table_rec.PROJECT_ID);
	    clob_result := REPLACE(clob_result, '<CONNECTION_ID>',  table_rec.CONNECTION_ID);
	    clob_result := REPLACE(clob_result, '<CATALOG_NAME>',  table_rec.CATALOG_NAME);
	    clob_result := REPLACE(clob_result, '<SOURCE_SCHEMA_NAME>',  table_rec.SCHEMA_NAME);
	    clob_result := REPLACE(clob_result, '<SOURCE_TABLE_NAME>',  table_rec.OBJECT_NAME);
	    clob_result := REPLACE(clob_result, '<SCHEMA_NAME>',  table_rec.SCHEMA_NAME);
	    clob_result := REPLACE(clob_result, '<OBJECT_NAME>',  table_rec.OBJECT_NAME);
	    clob_result := REPLACE(clob_result, '<CAPT_VERSION>',  table_rec.CAPT_VERSION);
	    clob_result := REPLACE(clob_result, '<CONV_VERSION>',  table_rec.CONV_VERSION);
	    clob_result := REPLACE(clob_result, '<GENERATE_VERSION>',  table_rec.GENERATE_VERSION);
	    clob_result := REPLACE(clob_result, '<SQL_SCRIPT>',  table_rec.sql_script);
	    clob_result := REPLACE(clob_result, '<START_DATE>',  start_date);
	    
        --dbms_output.put_line('clob_result -> ' || clob_result);        
        
        INSERT INTO MD_GENERATE(
                           PROJECT_ID,
                           CONNECTION_ID,
                           CATALOG_NAME,
                           SCHEMA_NAME,
                           CAPT_VERSION,
						   CONV_VERSION,
                           GENERATE_VERSION,
                           OBJECT_TYPE,
                           OBJECT_NAME,
                           SQL_SCRIPT,
                           CREATED_ON
                          )
                VALUES (
                           table_rec.PROJECT_ID,
                           table_rec.CONNECTION_ID,
                           table_rec.CATALOG_NAME,
                           table_rec.SCHEMA_NAME,
                           table_rec.CAPT_VERSION,
						   table_rec.CONV_VERSION,
                           table_rec.GENERATE_VERSION,
                           'JOB_SCHEDULER',
                           table_rec.OBJECT_NAME,
                           clob_result,
                           CURRENT_TIMESTAMP(3)
                );
       COMMIT;
        
   END LOOP;
END GenerateSchedulerJobs ;


-- Generated all TABLES using WITH CLAUSE for performance but took longer time 
-- 1. 'With Clause' option : PROCEDURE GenerateAllTables : : it took 119 sec
-- 2. Function option : PROCEDURE GenerateTables : MIGRATION_GENERATOR.generate_table : : it tool 93 sec
PROCEDURE GenerateAllTables(projectID MD_PROJECTS.PROJECT_ID%TYPE, connectionID MD_CONNECTIONS.CONNECTION_ID%TYPE, catalogName MD_CATALOGS.CATALOG_NAME%TYPE) 
IS
BEGIN

  INSERT INTO MD_GENERATE(
                           PROJECT_ID,
                           CONNECTION_ID,
                           CATALOG_NAME,
                           SCHEMA_NAME,
                           CAPT_VERSION,
						   CONV_VERSION,
                           GENERATE_VERSION,
                           OBJECT_TYPE,
                           OBJECT_NAME,
                           SQL_SCRIPT,
                           CREATED_ON
                          )                     
  WITH COLUMN_CLS AS (
   SELECT C.CATALOG_NAME, C.SCHEMA_NAME, C.TABLE_NAME,
   RTRIM(XMLAGG(XMLELEMENT(E, CHR(10) || '   ' ||
          C.COLUMN_NAME ||' ' ||
          C.COLUMN_TYPE ||
          DECODE(C.COLUMN_TYPE,'INT',null,'DATE',null,'TIMESTAMP',null,'LONG',null,'CLOB',null,'BLOB',null,'DECIMAL',null,'BINARY_DOUBLE',null,'NUMBER',DECODE(TO_CHAR(C.PRECISION),null,null,'('),'(') ||
          DECODE(C.COLUMN_TYPE,'INT',null,'DATE',null,'TIMESTAMP',null,'LONG',null,'CLOB',null,'BLOB',null,'DECIMAL',null,'BINARY_DOUBLE',null,'FLOAT',C.PRECISION,'CHAR', C.PRECISION ||' CHAR', 'CHARACTER', C.PRECISION ||' CHAR','VARCHAR2',C.PRECISION ||' CHAR', 'NCHAR',C.PRECISION,'NVARCHAR2',C.PRECISION,'NUMBER',DECODE(TO_CHAR(C.PRECISION),null,null, TO_CHAR(C.PRECISION) || ',' || TO_CHAR(C.SCALE)),  '******ERROR') ||
          DECODE(C.COLUMN_TYPE,'INT',null,'DATE',null,'TIMESTAMP',null,'LONG',null,'CLOB',null,'BLOB',null,'DECIMAL',null,'BINARY_DOUBLE',null,'NUMBER',DECODE(TO_CHAR(C.PRECISION),null,null,')'),')') ||
          DECODE(nullable,'N',' NOT NULL',null) ||
          ',')).EXTRACT('/E/text()').GETCLOBVAL(), ',') ||CHR(10)|| ');' tableDdl
      FROM
          MD_COLUMNS C
      WHERE
         C.PROJECT_ID = projectID
        AND C.CONNECTION_ID = connectionID
        AND C.TYPE = 'CONVERTED'
        AND C.CATALOG_NAME = catalogName
        GROUP BY C.CATALOG_NAME, C.SCHEMA_NAME, C.TABLE_NAME
  )
                    (SELECT T.PROJECT_ID,
                           T.CONNECTION_ID,
                           T.CATALOG_NAME,
                           T.SCHEMA_NAME,
                           g_captureVersionId,
						   T.CONV_VERSION,
						   g_generateVersionId,
                           'TABLE',
                           T.TABLE_NAME,
                           'PROMPT Create Table '|| T.TABLE_NAME || CHR(10)||
                           'CREATE TABLE ' || T.SCHEMA_NAME || '.' || T.TABLE_NAME || CHR(10)|| '('|| C.tableDdl,
                            CURRENT_TIMESTAMP(3)
                    FROM MD_TABLES T, COLUMN_CLS C
                    WHERE TYPE = 'CONVERTED'
                    AND T.ORACLE_NAME = C.TABLE_NAME
                    AND T.SCHEMA_NAME = C.SCHEMA_NAME
                    AND T.CATALOG_NAME = C.CATALOG_NAME
                    );


END GenerateAllTables;

PROCEDURE generate_tables(projectID IN NUMBER, connectionID IN NUMBER, sourceDatabase IN VARCHAR2, schema_array IN generateObjectArray DEFAULT NULL, table_array IN generateObjectArray DEFAULT NULL)  
IS
BEGIN
    g_startTime := CURRENT_TIMESTAMP;

    IF g_generateVersionId IS NULL or g_generateVersionId = 0 THEN
       GenerateVersionID(projectID, connectionID, sourceDatabase, 0);    
    END IF;

    DBMS_OUTPUT.PUT_LINE('IN TABLE');
    DBMS_OUTPUT.PUT_LINE('Captured Version for ' || sourceDatabase || ' is ' || g_captureVersionId);
	DBMS_OUTPUT.PUT_LINE('Convert Version for ' || sourceDatabase || ' is ' || g_convertVersionId);
    DBMS_OUTPUT.PUT_LINE('Generated Version for ' || sourceDatabase || ' is ' || g_generateVersionId);


     INSERT INTO MD_GENERATE(
                           PROJECT_ID,
                           CONNECTION_ID,
                           CATALOG_NAME,
                           SCHEMA_NAME,
                           CAPT_VERSION,
						   CONV_VERSION,
                           GENERATE_VERSION,
                           OBJECT_TYPE,
                           OBJECT_NAME,
                           SQL_SCRIPT,
                           CREATED_ON
                          )
                    (SELECT T.PROJECT_ID,
                            T.CONNECTION_ID,
                            T.CATALOG_NAME,
                            S.ORACLE_NAME,
                            g_captureVersionId,
							T.CONV_VERSION,
							g_generateVersionId,
                            'TABLE',
                            T.ORACLE_NAME,
                            MIGRATION_GENERATOR.gen_table_sql(
                                          T.PROJECT_ID,
                                          T.CONNECTION_ID,
                                          T.CATALOG_NAME,
                                          S.SCHEMA_NAME,
                                          S.ORACLE_NAME,
                                          T.TABLE_NAME,
                                          T.ORACLE_NAME
                                          ),
                            CURRENT_TIMESTAMP(3)
                    FROM MD_TABLES T, MD_SCHEMAS S
                    WHERE S.TYPE = 'CONVERTED'
                    AND T.TYPE = S.TYPE
                    AND T.PROJECT_ID = S.PROJECT_ID
                    AND T.CONNECTION_ID = S.CONNECTION_ID
                    AND T.CATALOG_NAME = S.CATALOG_NAME
                    AND T.SCHEMA_NAME = S.SCHEMA_NAME
                    AND ((projectID IS NULL) OR (S.PROJECT_ID = projectID)) 
                    AND ((connectionID IS NULL) OR (S.CONNECTION_ID = connectionID))
                    AND (((schema_array IS NULL) AND (table_array IS NULL)) 
                           OR  ((schema_array IS NULL) OR (S.SCHEMA_NAME IN (SELECT COLUMN_VALUE from TABLE(schema_array)))) 
                           AND ((table_array IS NULL) OR (T.TABLE_NAME IN (SELECT COLUMN_VALUE from TABLE(table_array))))
                        )
                    -- AND T.CAPT_VERSION = g_captureVersionId
                    AND S.CONV_VERSION = T.CONV_VERSION
      				AND T.CONV_VERSION = g_convertVersionId
                    );

    g_endTime := CURRENT_TIMESTAMP;
    DBMS_OUTPUT.PUT_LINE(' Migration Selected TABLES Generation Time: ' || TO_CHAR(g_endTime - g_startTime));
    commit;
END generate_tables;


PROCEDURE generate_table_defaults(projectID IN NUMBER, connectionID IN NUMBER, sourceDatabase IN VARCHAR2, schema_array IN generateObjectArray DEFAULT NULL, table_array IN generateObjectArray DEFAULT NULL) 
IS
BEGIN
     g_startTime := CURRENT_TIMESTAMP;

	IF g_generateVersionId IS NULL or g_generateVersionId = 0 THEN
       GenerateVersionID(projectID, connectionID, sourceDatabase, 0);    
    END IF;
    
     INSERT INTO MD_GENERATE(
                           PROJECT_ID,
                           CONNECTION_ID,
                           CATALOG_NAME,
                           SCHEMA_NAME,
                           CAPT_VERSION,
						   CONV_VERSION,
                           GENERATE_VERSION,
                           OBJECT_TYPE,
                           OBJECT_NAME,
                           SQL_SCRIPT,
                           CREATED_ON
                          )
                    (SELECT COL.PROJECT_ID,
                           COL.CONNECTION_ID,
                           COL.CATALOG_NAME,
                           COL.schemaOracleName,
                           g_captureVersionId,
						   COL.CONV_VERSION,
						   g_generateVersionId,
                           'TABLE_DEFAULT',
                           COL.tableOracleName,
                           MIGRATION_GENERATOR.gen_tabledefaults_sql(
                                          COL.PROJECT_ID,
                                          COL.CONNECTION_ID,
                                          COL.CATALOG_NAME,
                                          COL.SCHEMA_NAME,
                                          COL.schemaOracleName,
                                          COL.TABLE_NAME,
                                          COL.tableOracleName
                                          ),
                            CURRENT_TIMESTAMP(3)
                    FROM (
                      SELECT  DISTINCT C.PROJECT_ID, C.CONNECTION_ID, C.CATALOG_NAME, C.SCHEMA_NAME, C.CONV_VERSION, C.TABLE_NAME, S.ORACLE_NAME schemaOracleName, T.ORACLE_NAME tableOracleName
                      FROM
                          MD_COLUMNS C, MD_SCHEMAS S, MD_TABLES T
                      WHERE
                      C.DEFAULT_VALUE is NOT NULL
                      AND C.TYPE = 'CONVERTED' 
                      AND T.TYPE = C.TYPE
                      AND S.TYPE = C.TYPE
                      AND C.PROJECT_ID = S.PROJECT_ID
                      AND C.CONNECTION_ID = S.CONNECTION_ID
                      AND C.CATALOG_NAME = S.CATALOG_NAME
                      AND C.SCHEMA_NAME = S.SCHEMA_NAME                  
                      AND S.PROJECT_ID = T.PROJECT_ID
                      AND S.CONNECTION_ID = T.CONNECTION_ID
                      AND S.CATALOG_NAME = T.CATALOG_NAME
                      AND S.SCHEMA_NAME = T.SCHEMA_NAME
                      AND T.TABLE_NAME = C.TABLE_NAME                     
                      AND ((projectID IS NULL) OR (T.PROJECT_ID = projectID)) 
                      AND ((connectionID IS NULL) OR (T.CONNECTION_ID = connectionID))
                      AND (((schema_array IS NULL) AND (table_array IS NULL)) 
                           OR  ((schema_array IS NULL) OR (T.SCHEMA_NAME IN (SELECT COLUMN_VALUE from TABLE(schema_array)))) 
                           AND ((table_array IS NULL) OR (C.TABLE_NAME IN (SELECT COLUMN_VALUE from TABLE(table_array))))
                          ) 
                      -- AND T.CAPT_VERSION = g_captureVersionId
                      AND C.CONV_VERSION = S.CONV_VERSION
                      AND S.CONV_VERSION = T.CONV_VERSION
      				  AND T.CONV_VERSION = g_convertVersionId                    
                    ) COL );

    g_endTime := CURRENT_TIMESTAMP;
    DBMS_OUTPUT.PUT_LINE(' Migration Selected TABLE DEFAULTS Generation Time: ' || TO_CHAR(g_endTime - g_startTime));
    commit;
END generate_table_defaults;

-- Generate one selected PrimaryKey : in a project, connections, databases and given schema
PROCEDURE generate_pk_constraints(projectID IN NUMBER, connectionID IN NUMBER, sourceDatabase IN VARCHAR2, schema_array IN generateObjectArray DEFAULT NULL, table_array IN generateObjectArray DEFAULT NULL, primaryKey_array IN generateObjectArray DEFAULT NULL) 
IS
BEGIN
     g_startTime := CURRENT_TIMESTAMP;
     
	IF g_generateVersionId IS NULL or g_generateVersionId = 0 THEN
       GenerateVersionID(projectID, connectionID, sourceDatabase, 0);    
    END IF;
    
    INSERT INTO MD_GENERATE(
                           PROJECT_ID,
                           CONNECTION_ID,
                           CATALOG_NAME,
                           SCHEMA_NAME,
                           CAPT_VERSION,
						   CONV_VERSION,
                           GENERATE_VERSION,
                           OBJECT_TYPE,
                           OBJECT_NAME,
                           SQL_SCRIPT,
                           CREATED_ON
                          )
                   (
                    SELECT CON.PROJECT_ID,
                           CON.CONNECTION_ID,
                           CON.CATALOG_NAME,
                           CON.schemaOracleName,
                           g_captureVersionId,
						   CON.CONV_VERSION,
						   g_generateVersionId,
                           'PRIMARY_KEY',
                           CON.consOracleName,
                           MIGRATION_GENERATOR.gen_primaryKey_sql(
                                          CON.PROJECT_ID,
                                          CON.CONNECTION_ID,
                                          CON.CATALOG_NAME,
                                          CON.SCHEMA_NAME,
                                          CON.schemaOracleName,
                                          CON.TABLE_NAME,
                                          CON.tableOracleName,
                                          CON.CONSTRAINT_NAME),
                          CURRENT_TIMESTAMP(3)
                    FROM (
                      SELECT  DISTINCT C.PROJECT_ID, C.CONNECTION_ID, C.CATALOG_NAME, C.SCHEMA_NAME, C.CONV_VERSION, C.TABLE_NAME, C.CONSTRAINT_NAME, C.ORACLE_NAME consOracleName, S.ORACLE_NAME schemaOracleName, T.ORACLE_NAME tableOracleName
                      FROM
                          MD_CONSTRAINTS C, MD_SCHEMAS S, MD_TABLES T
                      WHERE
                      C.CONSTRAINT_TYPE = 'PK'
                      AND C.TYPE = 'CONVERTED' 
                      AND T.TYPE = C.TYPE
                      AND S.TYPE = C.TYPE
                      AND C.PROJECT_ID = S.PROJECT_ID
                      AND C.CONNECTION_ID = S.CONNECTION_ID
                      AND C.CATALOG_NAME = S.CATALOG_NAME
                      AND C.SCHEMA_NAME = S.SCHEMA_NAME                  
                      AND S.PROJECT_ID = T.PROJECT_ID
                      AND S.CONNECTION_ID = T.CONNECTION_ID
                      AND S.CATALOG_NAME = T.CATALOG_NAME
                      AND S.SCHEMA_NAME = T.SCHEMA_NAME
                      AND T.TABLE_NAME = C.TABLE_NAME
                      AND ((projectID IS NULL) OR (T.PROJECT_ID = projectID)) 
                      AND ((connectionID IS NULL) OR (T.CONNECTION_ID = connectionID))
                      AND (((schema_array IS NULL) AND (table_array IS NULL) AND (primaryKey_array IS NULL)) 
                           OR  ((schema_array IS NULL) OR (T.SCHEMA_NAME IN (SELECT COLUMN_VALUE from TABLE(schema_array)))) 
                           AND ((table_array IS NULL) OR (C.TABLE_NAME IN (SELECT COLUMN_VALUE from TABLE(table_array))))
                           AND ((primaryKey_array IS NULL) OR (C.CONSTRAINT_NAME IN (SELECT COLUMN_VALUE from TABLE(primaryKey_array))))
                        )
                      -- AND T.CAPT_VERSION = g_captureVersionId
                      AND C.CONV_VERSION = S.CONV_VERSION
                      AND S.CONV_VERSION = T.CONV_VERSION
      				  AND T.CONV_VERSION = g_convertVersionId  
                    ) CON );

    g_endTime := CURRENT_TIMESTAMP;
    DBMS_OUTPUT.PUT_LINE(' Migration Selected PRIMARY KEYS Generation Time: ' || TO_CHAR(g_endTime - g_startTime));

      EXCEPTION 
      WHEN DUP_VAL_ON_INDEX THEN
      DBMS_OUTPUT.PUT_LINE('PrimaryKeyConstraints : Cannot insert duplicate');
  --    DBMS_OUTPUT.PUT_LINE('Primary Keys -> ' || OBJECT_NAME);
     commit;                
END generate_pk_constraints;


-- Generate one selected ForeignKey : in a project, connections, databases and given schema
PROCEDURE generate_fk_constraints(projectID IN NUMBER, connectionID IN NUMBER, sourceDatabase IN VARCHAR2, schema_array IN generateObjectArray DEFAULT NULL, table_array IN generateObjectArray DEFAULT NULL, foreignKey_array IN generateObjectArray DEFAULT NULL) 
IS
BEGIN
     g_startTime := CURRENT_TIMESTAMP;
     
	IF g_generateVersionId IS NULL or g_generateVersionId = 0 THEN
       GenerateVersionID(projectID, connectionID, sourceDatabase, 0);    
    END IF;
    
    INSERT INTO MD_GENERATE(
                           PROJECT_ID,
                           CONNECTION_ID,
                           CATALOG_NAME,
                           SCHEMA_NAME,                         
                           CAPT_VERSION,
						   CONV_VERSION,
                           GENERATE_VERSION,
                           OBJECT_TYPE,
                           OBJECT_NAME,
                           SQL_SCRIPT,
                           CREATED_ON
                          )
                   (
                    SELECT CON.PROJECT_ID,
                           CON.CONNECTION_ID,
                           CON.CATALOG_NAME,
                           CON.schemaOracleName,
                           g_captureVersionId,
						   CON.CONV_VERSION,
						   g_generateVersionId,
                           'FOREIGN_KEY',
                           CON.consOracleName,
                           MIGRATION_GENERATOR.gen_foreignKey_sql(
                                          CON.PROJECT_ID,
                                          CON.CONNECTION_ID,
                                          CON.CATALOG_NAME,
                                          CON.SCHEMA_NAME,
                                          CON.schemaOracleName,
                                          CON.TABLE_NAME,
                                          CON.tableOracleName,
                                          CON.CONSTRAINT_NAME),
                          CURRENT_TIMESTAMP(3)
                    FROM (
                      SELECT  DISTINCT C.PROJECT_ID, C.CONNECTION_ID, C.CATALOG_NAME, C.SCHEMA_NAME, C.CONV_VERSION, C.TABLE_NAME, C.CONSTRAINT_NAME, C.ORACLE_NAME consOracleName, S.ORACLE_NAME schemaOracleName, T.ORACLE_NAME tableOracleName
                      FROM
                          MD_CONSTRAINTS C, MD_SCHEMAS S, MD_TABLES T
                      WHERE
                      C.CONSTRAINT_TYPE = 'FOREIGN KEY'
                      AND C.TYPE = 'CONVERTED' 
                      AND T.TYPE = C.TYPE
                      AND S.TYPE = C.TYPE
                      AND C.PROJECT_ID = S.PROJECT_ID
                      AND C.CONNECTION_ID = S.CONNECTION_ID
                      AND C.CATALOG_NAME = S.CATALOG_NAME
                      AND C.SCHEMA_NAME = S.SCHEMA_NAME                  
                      AND S.PROJECT_ID = T.PROJECT_ID
                      AND S.CONNECTION_ID = T.CONNECTION_ID
                      AND S.CATALOG_NAME = T.CATALOG_NAME
                      AND S.SCHEMA_NAME = T.SCHEMA_NAME
                      AND T.TABLE_NAME = C.TABLE_NAME                   
                      AND ((projectID IS NULL) OR (T.PROJECT_ID = projectID)) 
                      AND ((connectionID IS NULL) OR (T.CONNECTION_ID = connectionID))
                      AND (((schema_array IS NULL) AND (table_array IS NULL) AND (foreignKey_array IS NULL)) 
                           OR  ((schema_array IS NULL) OR (T.SCHEMA_NAME IN (SELECT COLUMN_VALUE from TABLE(schema_array)))) 
                           AND ((table_array IS NULL) OR (C.TABLE_NAME IN (SELECT COLUMN_VALUE from TABLE(table_array))))
                           AND ((foreignKey_array IS NULL) OR (C.CONSTRAINT_NAME IN (SELECT COLUMN_VALUE from TABLE(foreignKey_array))))
                           )
                      AND C.TABLE_NAME IN ( SELECT DISTINCT C.TABLE_NAME FROM MD_GENERATE GEN
                                   WHERE GEN.OBJECT_TYPE = 'TABLE'
                                   AND S.ORACLE_NAME = GEN.SCHEMA_NAME
                                   AND C.CATALOG_NAME = GEN.CATALOG_NAME
                                   AND C.CONNECTION_ID = GEN.CONNECTION_ID
                                   AND C.PROJECT_ID = GEN.PROJECT_ID
                                   AND C.TABLE_NAME = GEN.OBJECT_NAME
                                   AND C.TYPE = 'CONVERTED'
                                   AND C.CONSTRAINT_TYPE = 'FOREIGN KEY'
                                   AND S.SCHEMA_NAME = C.SCHEMA_NAME
                                   AND S.CATALOG_NAME = GEN.CATALOG_NAME
                                   AND S.CONNECTION_ID = GEN.CONNECTION_ID
                                   AND S.PROJECT_ID = GEN.PROJECT_ID
                          )
                      AND C.REFTABLE_NAME IN ( SELECT DISTINCT C.REFTABLE_NAME FROM MD_GENERATE GEN
                                   WHERE GEN.OBJECT_TYPE = 'TABLE'
                                   AND S.ORACLE_NAME = GEN.SCHEMA_NAME
                                   AND C.REFCATALOG_NAME = GEN.CATALOG_NAME
                                   AND C.CONNECTION_ID = GEN.CONNECTION_ID
                                   AND C.PROJECT_ID = GEN.PROJECT_ID
                                   AND C.REFTABLE_NAME = GEN.OBJECT_NAME
                                   AND C.TYPE = 'CONVERTED'
                                   AND C.CONSTRAINT_TYPE = 'FOREIGN KEY'
                                   AND S.SCHEMA_NAME = C.REFSCHEMA_NAME
                                   AND S.CATALOG_NAME = GEN.CATALOG_NAME
                                   AND S.CONNECTION_ID = GEN.CONNECTION_ID
                                   AND S.PROJECT_ID = GEN.PROJECT_ID
                            )
                      -- AND T.CAPT_VERSION = g_captureVersionId
                      AND C.CONV_VERSION = S.CONV_VERSION
                      AND S.CONV_VERSION = T.CONV_VERSION
      				  AND T.CONV_VERSION = g_convertVersionId     
                    ) CON
                    );

    g_endTime := CURRENT_TIMESTAMP;
    DBMS_OUTPUT.PUT_LINE(' Migration Selected FOREIGN KEYS Generation Time: ' || TO_CHAR(g_endTime - g_startTime));
    commit;            
END generate_fk_constraints;

-- Generate one selected UniqueKey : in a project, connections, databases and given schema
PROCEDURE generate_uk_constraints(projectID IN NUMBER, connectionID IN NUMBER, sourceDatabase IN VARCHAR2, schema_array IN generateObjectArray DEFAULT NULL, table_array IN generateObjectArray DEFAULT NULL, uniqueKey_array IN generateObjectArray DEFAULT NULL) 
IS
BEGIN
     g_startTime := CURRENT_TIMESTAMP;
     
	IF g_generateVersionId IS NULL or g_generateVersionId = 0 THEN
       GenerateVersionID(projectID, connectionID, sourceDatabase, 0);    
    END IF;
    
    INSERT INTO MD_GENERATE(
                           PROJECT_ID,
                           CONNECTION_ID,
                           CATALOG_NAME,
                           SCHEMA_NAME,
                           CAPT_VERSION,
						   CONV_VERSION,
                           GENERATE_VERSION,
                           OBJECT_TYPE,
                           OBJECT_NAME,
                           SQL_SCRIPT,
                           CREATED_ON
                          )
                   (
                    SELECT CON.PROJECT_ID,
                           CON.CONNECTION_ID,
                           CON.CATALOG_NAME,
                           CON.schemaOracleName,
                           g_captureVersionId,
						   CON.CONV_VERSION,
						   g_generateVersionId,
                           'UNIQUE_KEY',
                           CON.consOracleName,
                           MIGRATION_GENERATOR.gen_uniqueKey_sql(
                                          CON.PROJECT_ID,
                                          CON.CONNECTION_ID,
                                          CON.CATALOG_NAME,
                                          CON.SCHEMA_NAME,
                                          CON.schemaOracleName,
                                          CON.TABLE_NAME,
                                          CON.tableOracleName,
                                          CON.CONSTRAINT_NAME),
                          CURRENT_TIMESTAMP(3)
                    FROM (
                      SELECT  DISTINCT C.PROJECT_ID, C.CONNECTION_ID, C.CATALOG_NAME, C.SCHEMA_NAME, C.CONV_VERSION, C.TABLE_NAME, C.CONSTRAINT_NAME, C.ORACLE_NAME consOracleName, S.ORACLE_NAME schemaOracleName, T.ORACLE_NAME tableOracleName
                      FROM
                          MD_CONSTRAINTS C, MD_SCHEMAS S, MD_TABLES T
                      WHERE
                      C.CONSTRAINT_TYPE = 'UNIQUE'
                      AND C.TYPE = 'CONVERTED' 
                      AND T.TYPE = C.TYPE
                      AND S.TYPE = C.TYPE
                      AND C.PROJECT_ID = S.PROJECT_ID
                      AND C.CONNECTION_ID = S.CONNECTION_ID
                      AND C.CATALOG_NAME = S.CATALOG_NAME
                      AND C.SCHEMA_NAME = S.SCHEMA_NAME                  
                      AND S.PROJECT_ID = T.PROJECT_ID
                      AND S.CONNECTION_ID = T.CONNECTION_ID
                      AND S.CATALOG_NAME = T.CATALOG_NAME
                      AND S.SCHEMA_NAME = T.SCHEMA_NAME
                      AND T.TABLE_NAME = C.TABLE_NAME
                      AND ((projectID IS NULL) OR (T.PROJECT_ID = projectID)) 
                      AND ((connectionID IS NULL) OR (T.CONNECTION_ID = connectionID))
                      AND (((schema_array IS NULL) AND (table_array IS NULL) AND (uniqueKey_array IS NULL)) 
                           OR  ((schema_array IS NULL) OR (T.SCHEMA_NAME IN (SELECT COLUMN_VALUE from TABLE(schema_array)))) 
                           AND ((table_array IS NULL) OR (C.TABLE_NAME IN (SELECT COLUMN_VALUE from TABLE(table_array))))
                           AND ((uniqueKey_array IS NULL) OR (C.CONSTRAINT_NAME IN (SELECT COLUMN_VALUE from TABLE(uniqueKey_array))))
                          )
                      -- AND T.CAPT_VERSION = g_captureVersionId
                      AND C.CONV_VERSION = S.CONV_VERSION
                      AND S.CONV_VERSION = T.CONV_VERSION
      				  AND T.CONV_VERSION = g_convertVersionId  
                    ) CON );

    g_endTime := CURRENT_TIMESTAMP;
    DBMS_OUTPUT.PUT_LINE(' Migration selected UNIQUE KEYS Generation Time: ' || TO_CHAR(g_endTime - g_startTime));
    commit;
END generate_uk_constraints;

-- Generate one selected Check Constraints : in a project, connections, databases and given schema
PROCEDURE generate_check_constraints(projectID IN NUMBER, connectionID IN NUMBER, sourceDatabase IN VARCHAR2, schema_array IN generateObjectArray DEFAULT NULL, table_array IN generateObjectArray DEFAULT NULL, checkConstraint_array IN generateObjectArray DEFAULT NULL) 
IS
BEGIN
     g_startTime := CURRENT_TIMESTAMP;
     
	IF g_generateVersionId IS NULL or g_generateVersionId = 0 THEN
       GenerateVersionID(projectID, connectionID, sourceDatabase, 0);    
    END IF;
    
    INSERT INTO MD_GENERATE(
                           PROJECT_ID,
                           CONNECTION_ID,
                           CATALOG_NAME,
                           SCHEMA_NAME,
                           CAPT_VERSION,
						   CONV_VERSION,
                           GENERATE_VERSION,
                           OBJECT_TYPE,
                           OBJECT_NAME,
                           SQL_SCRIPT,
                           CREATED_ON
                          )
                   (
                    SELECT CON.PROJECT_ID,
                           CON.CONNECTION_ID,
                           CON.CATALOG_NAME,
                           CON.schemaOracleName,
                           g_captureVersionId,
						   CON.CONV_VERSION,
						   g_generateVersionId,
                           'CHECK',
                           CON.consOracleName,
                           MIGRATION_GENERATOR.gen_checkConstraint_sql(
                                          CON.PROJECT_ID,
                                          CON.CONNECTION_ID,
                                          CON.CATALOG_NAME,
                                          CON.SCHEMA_NAME,
                                          CON.schemaOracleName,
                                          CON.TABLE_NAME,
                                          CON.tableOracleName,
                                          CON.CONSTRAINT_NAME),
                          CURRENT_TIMESTAMP(3)
                    FROM (
                      SELECT  DISTINCT C.PROJECT_ID, C.CONNECTION_ID, C.CATALOG_NAME, C.SCHEMA_NAME, C.CONV_VERSION, C.TABLE_NAME, C.CONSTRAINT_NAME, C.ORACLE_NAME consOracleName, S.ORACLE_NAME schemaOracleName, T.ORACLE_NAME tableOracleName
                      FROM
                          MD_CONSTRAINTS C, MD_SCHEMAS S, MD_TABLES T
                      WHERE
                      C.CONSTRAINT_TYPE = 'CHECK'
                      AND C.TYPE = 'CONVERTED' 
                      AND T.TYPE = C.TYPE
                      AND S.TYPE = C.TYPE
                      AND C.PROJECT_ID = S.PROJECT_ID
                      AND C.CONNECTION_ID = S.CONNECTION_ID
                      AND C.CATALOG_NAME = S.CATALOG_NAME
                      AND C.SCHEMA_NAME = S.SCHEMA_NAME                  
                      AND S.PROJECT_ID = T.PROJECT_ID
                      AND S.CONNECTION_ID = T.CONNECTION_ID
                      AND S.CATALOG_NAME = T.CATALOG_NAME
                      AND S.SCHEMA_NAME = T.SCHEMA_NAME
                      AND T.TABLE_NAME = C.TABLE_NAME
                      AND ((projectID IS NULL) OR (T.PROJECT_ID = projectID)) 
                      AND ((connectionID IS NULL) OR (T.CONNECTION_ID = connectionID))
                      AND (((schema_array IS NULL) AND (table_array IS NULL) AND (checkConstraint_array IS NULL)) 
                           OR  ((schema_array IS NULL) OR (T.SCHEMA_NAME IN (SELECT COLUMN_VALUE from TABLE(schema_array)))) 
                           AND ((table_array IS NULL) OR (C.TABLE_NAME IN (SELECT COLUMN_VALUE from TABLE(table_array))))
                           AND ((checkConstraint_array IS NULL) OR (C.CONSTRAINT_NAME IN (SELECT COLUMN_VALUE from TABLE(checkConstraint_array))))
                          )
                      -- AND T.CAPT_VERSION = g_captureVersionId
                      AND C.CONV_VERSION = S.CONV_VERSION
                      AND S.CONV_VERSION = T.CONV_VERSION
      				  AND T.CONV_VERSION = g_convertVersionId  
                    ) CON );

    g_endTime := CURRENT_TIMESTAMP;
    DBMS_OUTPUT.PUT_LINE(' Migration selected CHECK CONSTRAINTS Generation Time: ' || TO_CHAR(g_endTime - g_startTime));
    commit;
END generate_check_constraints;

-- Generate one selected Indexes : in a project, connections, databases and given schema
PROCEDURE generate_indexes(projectID IN NUMBER, connectionID IN NUMBER, sourceDatabase IN VARCHAR2, schema_array IN generateObjectArray DEFAULT NULL, table_array IN generateObjectArray DEFAULT NULL, index_array IN generateObjectArray DEFAULT NULL) 
IS
BEGIN
     g_startTime := CURRENT_TIMESTAMP;
     
	IF g_generateVersionId IS NULL or g_generateVersionId = 0 THEN
       GenerateVersionID(projectID, connectionID, sourceDatabase, 0);    
    END IF;
    
	  INSERT INTO MD_GENERATE(
                           PROJECT_ID,
                           CONNECTION_ID,
                           CATALOG_NAME,
                           SCHEMA_NAME,
                           CAPT_VERSION,
						   CONV_VERSION,
                           GENERATE_VERSION,
                           OBJECT_TYPE,
                           OBJECT_NAME,
                           SQL_SCRIPT,
                           CREATED_ON
                          )
                   (
                    SELECT I.PROJECT_ID,
                           I.CONNECTION_ID,
                           I.CATALOG_NAME,
                           I.schemaOracleName,
                           g_captureVersionId,
						   I.CONV_VERSION,
						   g_generateVersionId,
                           'INDEX',
                           I.indxOracleName,
                           MIGRATION_GENERATOR.gen_index_sql(
                                          I.PROJECT_ID,
                                          I.CONNECTION_ID,
                                          I.CATALOG_NAME,
                                          I.SCHEMA_NAME,
                                          I.schemaOracleName,
                                          I.TABLE_NAME,
                                          I.tableOracleName,
                                          I.INDEX_NAME),
                          CURRENT_TIMESTAMP(3)
                    FROM (
                      SELECT  DISTINCT C.PROJECT_ID, C.CONNECTION_ID, C.CATALOG_NAME, C.SCHEMA_NAME, C.CONV_VERSION, C.TABLE_NAME, C.INDEX_NAME, C.ORACLE_NAME indxOracleName, S.ORACLE_NAME schemaOracleName, T.ORACLE_NAME tableOracleName
                      FROM
                          MD_INDEXES C, MD_SCHEMAS S, MD_TABLES T
                      WHERE
                      C.TYPE = 'CONVERTED' 
                      AND T.TYPE = C.TYPE
                      AND S.TYPE = C.TYPE
                      AND C.PROJECT_ID = S.PROJECT_ID
                      AND C.CONNECTION_ID = S.CONNECTION_ID
                      AND C.CATALOG_NAME = S.CATALOG_NAME
                      AND C.SCHEMA_NAME = S.SCHEMA_NAME                  
                      AND S.PROJECT_ID = T.PROJECT_ID
                      AND S.CONNECTION_ID = T.CONNECTION_ID
                      AND S.CATALOG_NAME = T.CATALOG_NAME
                      AND S.SCHEMA_NAME = T.SCHEMA_NAME
                      AND T.TABLE_NAME = C.TABLE_NAME
                      AND ((projectID IS NULL) OR (T.PROJECT_ID = projectID)) 
                      AND ((connectionID IS NULL) OR (T.CONNECTION_ID = connectionID))
                      AND (((schema_array IS NULL) AND (table_array IS NULL) AND (index_array IS NULL)) 
                           OR  ((schema_array IS NULL) OR (T.SCHEMA_NAME IN (SELECT COLUMN_VALUE from TABLE(schema_array)))) 
                           AND ((table_array IS NULL) OR (C.TABLE_NAME IN (SELECT COLUMN_VALUE from TABLE(table_array))))
                           AND ((index_array IS NULL) OR (C.INDEX_NAME IN (SELECT COLUMN_VALUE from TABLE(index_array))))
                          )
                      -- AND T.CAPT_VERSION = g_captureVersionId
                      AND C.CONV_VERSION = S.CONV_VERSION
                      AND S.CONV_VERSION = T.CONV_VERSION
      				  AND T.CONV_VERSION = g_convertVersionId  
                    ) I );


    g_endTime := CURRENT_TIMESTAMP;
    DBMS_OUTPUT.PUT_LINE(' Migration Selected INDEXES (Unique and Non Unique) Generation Time: ' || TO_CHAR(g_endTime - g_startTime));
    commit;                
END generate_indexes;

-- Generate one selected Sequence : in a project, connections, databases and given schema
PROCEDURE generate_sequences(projectID IN NUMBER, connectionID IN NUMBER, sourceDatabase IN VARCHAR2, schema_array IN generateObjectArray DEFAULT NULL, sequence_array IN generateObjectArray DEFAULT NULL) 
IS
BEGIN
    g_startTime := CURRENT_TIMESTAMP;
    
	IF g_generateVersionId IS NULL or g_generateVersionId = 0 THEN
       GenerateVersionID(projectID, connectionID, sourceDatabase, 0);    
    END IF;
    
        INSERT INTO MD_GENERATE(
                           PROJECT_ID,
                           CONNECTION_ID,
                           CATALOG_NAME,
                           SCHEMA_NAME,
                           CAPT_VERSION,
						   CONV_VERSION,
                           GENERATE_VERSION,
                           OBJECT_TYPE,
                           OBJECT_NAME,
                           SQL_SCRIPT,
                           CREATED_ON
                          )
                    (SELECT T.PROJECT_ID,
                            T.CONNECTION_ID,
                            T.CATALOG_NAME,
                            S.ORACLE_NAME,
                            g_captureVersionId,
						    T.CONV_VERSION,
						    g_generateVersionId,
                            'SEQUENCE',
                            T.ORACLE_NAME,
                            MIGRATION_GENERATOR.gen_sequence_sql(
                                          T.PROJECT_ID,
                                          T.CONNECTION_ID,
                                          T.CATALOG_NAME,
                                          S.SCHEMA_NAME,
                                          S.ORACLE_NAME,
                                          T.SEQUENCE_NAME,
                                          T.ORACLE_NAME
                                          ),
                            CURRENT_TIMESTAMP(3)
                    FROM MD_SEQUENCES T, MD_SCHEMAS S
                    WHERE S.TYPE = 'CONVERTED'
                    AND T.TYPE = S.TYPE
                    AND T.PROJECT_ID = S.PROJECT_ID
                    AND T.CONNECTION_ID = S.CONNECTION_ID
                    AND T.CATALOG_NAME = S.CATALOG_NAME
                    AND T.SCHEMA_NAME = S.SCHEMA_NAME
                    AND ((projectID IS NULL) OR (S.PROJECT_ID = projectID)) 
                    AND ((connectionID IS NULL) OR (S.CONNECTION_ID = connectionID))
                    AND (((schema_array IS NULL) AND (sequence_array IS NULL)) 
                           OR  ((schema_array IS NULL) OR (S.SCHEMA_NAME IN (SELECT COLUMN_VALUE from TABLE(schema_array)))) 
                           AND ((sequence_array IS NULL) OR (T.SEQUENCE_NAME IN (SELECT COLUMN_VALUE from TABLE(sequence_array))))
                        )
                    -- AND T.CAPT_VERSION = g_captureVersionId
                    AND S.CONV_VERSION = T.CONV_VERSION
      				AND T.CONV_VERSION = g_convertVersionId  
                    );

    g_endTime := CURRENT_TIMESTAMP;
    DBMS_OUTPUT.PUT_LINE(' Migration Selected SEQUENCES Generation Time: ' || TO_CHAR(g_endTime - g_startTime));
    commit;
END generate_sequences;

-- Generate one selected Triggers : in a project, connections, databases and given schema
PROCEDURE generate_triggers(projectID IN NUMBER, connectionID IN NUMBER, sourceDatabase IN VARCHAR2, schema_array IN generateObjectArray DEFAULT NULL, trigger_array IN generateObjectArray DEFAULT NULL) 
IS
BEGIN
     g_startTime := CURRENT_TIMESTAMP;
     
	IF g_generateVersionId IS NULL or g_generateVersionId = 0 THEN
       GenerateVersionID(projectID, connectionID, sourceDatabase, 0);    
    END IF;
    
     INSERT INTO MD_GENERATE(
                           PROJECT_ID,
                           CONNECTION_ID,
                           CATALOG_NAME,
                           SCHEMA_NAME,
                           CAPT_VERSION,
						   CONV_VERSION,
                           GENERATE_VERSION,
                           OBJECT_TYPE,
                           OBJECT_NAME,
                           SQL_SCRIPT,
                           CREATED_ON
                          )
                    (SELECT T.PROJECT_ID,
                           T.CONNECTION_ID,
                           T.CATALOG_NAME,
                           S.ORACLE_NAME,
                           g_captureVersionId,
						   T.CONV_VERSION,
						   g_generateVersionId,
                           'TRIGGER',
                           T.ORACLE_NAME,
                           T.NATIVE_SQL,
                           CURRENT_TIMESTAMP(3)
                    FROM MD_TRIGGERS T, MD_SCHEMAS S
                    WHERE S.TYPE = 'CONVERTED'
                    AND T.TYPE = S.TYPE
                    AND T.PROJECT_ID = S.PROJECT_ID
                    AND T.CONNECTION_ID = S.CONNECTION_ID
                    AND T.CATALOG_NAME = S.CATALOG_NAME
                    AND T.SCHEMA_NAME = S.SCHEMA_NAME
                    AND ((projectID IS NULL) OR (S.PROJECT_ID = projectID)) 
                    AND ((connectionID IS NULL) OR (S.CONNECTION_ID = connectionID))
                    AND (((schema_array IS NULL) AND (trigger_array IS NULL)) 
                           OR  ((schema_array IS NULL) OR (S.SCHEMA_NAME IN (SELECT COLUMN_VALUE from TABLE(schema_array)))) 
                           AND ((trigger_array IS NULL) OR (T.TRIGGER_NAME IN (SELECT COLUMN_VALUE from TABLE(trigger_array))))
                        )
                    -- AND T.CAPT_VERSION = g_captureVersionId
                    AND S.CONV_VERSION = T.CONV_VERSION
      				AND T.CONV_VERSION = g_convertVersionId  
                    );

    g_endTime := CURRENT_TIMESTAMP;
    DBMS_OUTPUT.PUT_LINE(' Migration Selected TRIGGERS Generation Time: ' || TO_CHAR(g_endTime - g_startTime));
    commit;                    
END generate_triggers;

-- Generate one selected StoredPrograms : in a project, connections, databases and given schema
PROCEDURE generate_stored_programs(projectID IN NUMBER, connectionID IN NUMBER, sourceDatabase IN VARCHAR2, schema_array IN generateObjectArray DEFAULT NULL, storedProgram_array IN generateObjectArray DEFAULT NULL) 
IS
BEGIN
     g_startTime := CURRENT_TIMESTAMP;
     
	IF g_generateVersionId IS NULL or g_generateVersionId = 0 THEN
       GenerateVersionID(projectID, connectionID, sourceDatabase, 0);    
    END IF;
    
     INSERT INTO MD_GENERATE(
                           PROJECT_ID,
                           CONNECTION_ID,
                           CATALOG_NAME,
                           SCHEMA_NAME,
                           CAPT_VERSION,
						   CONV_VERSION,
                           GENERATE_VERSION,
                           OBJECT_TYPE,
                           OBJECT_NAME,
                           SQL_SCRIPT,
                           CREATED_ON
                          )
                    (SELECT P.PROJECT_ID,
                           P.CONNECTION_ID,
                           P.CATALOG_NAME,
                           S.ORACLE_NAME,
                           g_captureVersionId,
						   P.CONV_VERSION,
						   g_generateVersionId,
                           'STORED_PROGRAM',
                           P.ORACLE_NAME,
                           P.NATIVE_SQL,
                            CURRENT_TIMESTAMP(3)
                    FROM MD_STORED_PROGRAMS P, MD_SCHEMAS S
                    WHERE S.TYPE = 'CONVERTED'
                    AND P.TYPE = S.TYPE
                    AND P.PROJECT_ID = S.PROJECT_ID
                    AND P.CONNECTION_ID = S.CONNECTION_ID
                    AND P.CATALOG_NAME = S.CATALOG_NAME
                    AND P.SCHEMA_NAME = S.SCHEMA_NAME
                    AND ((projectID IS NULL) OR (S.PROJECT_ID = projectID)) 
                    AND ((connectionID IS NULL) OR (S.CONNECTION_ID = connectionID))
                    AND (((schema_array IS NULL) AND (storedProgram_array IS NULL)) 
                           OR  ((schema_array IS NULL) OR (S.SCHEMA_NAME IN (SELECT COLUMN_VALUE from TABLE(schema_array)))) 
                           AND ((storedProgram_array IS NULL) OR (P.PROGRAM_NAME IN (SELECT COLUMN_VALUE from TABLE(storedProgram_array))))
                        )
                    -- AND T.CAPT_VERSION = g_captureVersionId
                    AND S.CONV_VERSION = P.CONV_VERSION
      				AND P.CONV_VERSION = g_convertVersionId  
                    );

    g_endTime := CURRENT_TIMESTAMP;
    DBMS_OUTPUT.PUT_LINE(' Migration Selected STORED PROGRAMS (Functions and Procedures) Generation Time: ' || TO_CHAR(g_endTime - g_startTime));
    commit;                   
END generate_stored_programs;

-- RUPPALA : 06/03/2014
-- OBSERVATION : On Teradata Claity DB for 132,012 Views, it took 21 Min to load all clobs using MIGRATION_GENERATOR.generate_views() function.
--               Direct INSERT INTO () SELECT NATIVE_SQL clob took only 5 Sec.
--               Conclusion : to copy clob objects from MD_* table to MD_GENRATE tables do not call 'MIGRATION_GENERATOR.generate_'
--               Also do not manipluate (add to remove strings) CLOB object during generation phase.

-- Generate one selected Views : in a project, connections, databases and given schema
PROCEDURE generate_views(projectID IN NUMBER, connectionID IN NUMBER, sourceDatabase IN VARCHAR2, schema_array IN generateObjectArray DEFAULT NULL, view_array IN generateObjectArray DEFAULT NULL) 
IS
BEGIN
     g_startTime := CURRENT_TIMESTAMP;
     
	IF g_generateVersionId IS NULL or g_generateVersionId = 0 THEN
       GenerateVersionID(projectID, connectionID, sourceDatabase, 0);    
    END IF;
    
        INSERT INTO MD_GENERATE(
                           PROJECT_ID,
                           CONNECTION_ID,
                           CATALOG_NAME,
                           SCHEMA_NAME,
                           CAPT_VERSION,
						   CONV_VERSION,
                           GENERATE_VERSION,
                           OBJECT_TYPE,
                           OBJECT_NAME,
                           SQL_SCRIPT,
                           CREATED_ON
                          )
                    (SELECT V.PROJECT_ID,
                           V.CONNECTION_ID,
                           V.CATALOG_NAME,
                           S.ORACLE_NAME,
                           g_captureVersionId,
						   V.CONV_VERSION,
						   g_generateVersionId,
                           'VIEW',
                           V.ORACLE_NAME,
                           V.NATIVE_SQL,
                           CURRENT_TIMESTAMP(3)
                    FROM MD_VIEWS V, MD_SCHEMAS S
                    WHERE S.TYPE = 'CONVERTED'
                    AND V.TYPE = S.TYPE
                    AND V.PROJECT_ID = S.PROJECT_ID
                    AND V.CONNECTION_ID = S.CONNECTION_ID
                    AND V.CATALOG_NAME = S.CATALOG_NAME
                    AND V.SCHEMA_NAME = S.SCHEMA_NAME
                    AND ((projectID IS NULL) OR (S.PROJECT_ID = projectID)) 
                    AND ((connectionID IS NULL) OR (S.CONNECTION_ID = connectionID))
                    AND (((schema_array IS NULL) AND (view_array IS NULL)) 
                           OR  ((schema_array IS NULL) OR (S.SCHEMA_NAME IN (SELECT COLUMN_VALUE from TABLE(schema_array)))) 
                           AND ((view_array IS NULL) OR (V.VIEW_NAME IN (SELECT COLUMN_VALUE from TABLE(view_array))))
                        )
                    -- AND T.CAPT_VERSION = g_captureVersionId
                    AND S.CONV_VERSION = V.CONV_VERSION
      				AND V.CONV_VERSION = g_convertVersionId  
                    );

    g_endTime := CURRENT_TIMESTAMP;
    DBMS_OUTPUT.PUT_LINE(' Migration Selected VIEWS Generation Time: ' || TO_CHAR(g_endTime - g_startTime));
    commit;
END generate_views;

-- Generate selected Packages in given table array : in a project, connections, databases and given schema
PROCEDURE generate_packages(projectID IN NUMBER, connectionID IN NUMBER, sourceDatabase IN VARCHAR2, schema_array IN generateObjectArray DEFAULT NULL, package_array IN generateObjectArray DEFAULT NULL) 
IS
BEGIN
     g_startTime := CURRENT_TIMESTAMP;
     
	IF g_generateVersionId IS NULL or g_generateVersionId = 0 THEN
       GenerateVersionID(projectID, connectionID, sourceDatabase, 0);    
    END IF;
    
-- Package Spec Clob insert
     INSERT INTO MD_GENERATE(
                           PROJECT_ID,
                           CONNECTION_ID,
                           CATALOG_NAME,
                           SCHEMA_NAME,
                           CAPT_VERSION,
						   CONV_VERSION,
                           GENERATE_VERSION,
                           OBJECT_TYPE,
                           OBJECT_NAME,
                           SQL_SCRIPT,
                           CREATED_ON
                          )
                    (SELECT P.PROJECT_ID,
                           P.CONNECTION_ID,
                           P.CATALOG_NAME,
                           S.ORACLE_NAME,
                           g_captureVersionId,
						   P.CONV_VERSION,
						   g_generateVersionId,
                           'PACKAGE_SPEC',
                           P.ORACLE_NAME,
                           P.PACKAGE_HEADER,
                           CURRENT_TIMESTAMP(3)
                    FROM MD_PACKAGES P, MD_SCHEMAS S
                    WHERE S.TYPE = 'CONVERTED'
                    AND P.TYPE = S.TYPE
                    AND P.PROJECT_ID = S.PROJECT_ID
                    AND P.CONNECTION_ID = S.CONNECTION_ID
                    AND P.CATALOG_NAME = S.CATALOG_NAME
                    AND P.SCHEMA_NAME = S.SCHEMA_NAME
                    AND ((projectID IS NULL) OR (S.PROJECT_ID = projectID)) 
                    AND ((connectionID IS NULL) OR (S.CONNECTION_ID = connectionID))
                    AND (((schema_array IS NULL) AND (package_array IS NULL)) 
                           OR  ((schema_array IS NULL) OR (S.SCHEMA_NAME IN (SELECT COLUMN_VALUE from TABLE(schema_array)))) 
                           AND ((package_array IS NULL) OR (P.PACKAGE_NAME IN (SELECT COLUMN_VALUE from TABLE(package_array))))
                        )
                    -- AND T.CAPT_VERSION = g_captureVersionId
                    AND S.CONV_VERSION = P.CONV_VERSION
      				AND P.CONV_VERSION = g_convertVersionId  
                    );

        commit; 
    
        -- Package Body Clob insert
        INSERT INTO MD_GENERATE(
                           PROJECT_ID,
                           CONNECTION_ID,
                           CATALOG_NAME,
                           SCHEMA_NAME,
                           CAPT_VERSION,
						   CONV_VERSION,
                           GENERATE_VERSION,
                           OBJECT_TYPE,
                           OBJECT_NAME,
                           SQL_SCRIPT,
                           CREATED_ON
                          )
                    (SELECT P.PROJECT_ID,
                           P.CONNECTION_ID,
                           P.CATALOG_NAME,                     
                           S.ORACLE_NAME,
                           g_captureVersionId,
						   P.CONV_VERSION,
						   g_generateVersionId,
                           'PACKAGE_BODY',
                           P.ORACLE_NAME,
                           P.NATIVE_SQL,
                           CURRENT_TIMESTAMP(3)
                    FROM MD_PACKAGES P, MD_SCHEMAS S
                    WHERE S.TYPE = 'CONVERTED'
                    AND P.TYPE = S.TYPE
                    AND P.PROJECT_ID = S.PROJECT_ID
                    AND P.CONNECTION_ID = S.CONNECTION_ID
                    AND P.CATALOG_NAME = S.CATALOG_NAME
                    AND P.SCHEMA_NAME = S.SCHEMA_NAME
                    AND ((projectID IS NULL) OR (S.PROJECT_ID = projectID)) 
                    AND ((connectionID IS NULL) OR (S.CONNECTION_ID = connectionID))
                    AND (((schema_array IS NULL) AND (package_array IS NULL)) 
                           OR  ((schema_array IS NULL) OR (S.SCHEMA_NAME IN (SELECT COLUMN_VALUE from TABLE(schema_array)))) 
                           AND ((package_array IS NULL) OR (P.PACKAGE_NAME IN (SELECT COLUMN_VALUE from TABLE(package_array))))
                        )
                     -- AND T.CAPT_VERSION = g_captureVersionId
                    AND S.CONV_VERSION = P.CONV_VERSION
      				AND P.CONV_VERSION = g_convertVersionId  
                    );

    g_endTime := CURRENT_TIMESTAMP;
    DBMS_OUTPUT.PUT_LINE(' Migration Selected PACKAGES Generation Time: ' || TO_CHAR(g_endTime - g_startTime));
    commit;                   
END generate_packages;

-- Generate one selected Synonyms : in a project, connections, databases and given schema
PROCEDURE generate_synonyms(projectID IN NUMBER, connectionID IN NUMBER, sourceDatabase IN VARCHAR2, schema_array IN generateObjectArray DEFAULT NULL, synonym_array IN generateObjectArray DEFAULT NULL) 
IS
BEGIN
     g_startTime := CURRENT_TIMESTAMP;
     
	IF g_generateVersionId IS NULL or g_generateVersionId = 0 THEN
       GenerateVersionID(projectID, connectionID, sourceDatabase, 0);    
    END IF;
    
    INSERT INTO MD_GENERATE(
                           PROJECT_ID,
                           CONNECTION_ID,
                           CATALOG_NAME,
                           SCHEMA_NAME,
                           CAPT_VERSION,
						   CONV_VERSION,
                           GENERATE_VERSION,
                           OBJECT_TYPE,
                           OBJECT_NAME,
                           SQL_SCRIPT,
                           CREATED_ON
                          )
                    (SELECT T.PROJECT_ID,
                            T.CONNECTION_ID,
                            T.CATALOG_NAME,
                            S.ORACLE_NAME,
                            g_captureVersionId,
						    S.CONV_VERSION,
						    g_generateVersionId,
                            'SYNONYM',
                            T.ORACLE_NAME,
                            MIGRATION_GENERATOR.gen_synonym_sql(
                                          T.PROJECT_ID,
                                          T.CONNECTION_ID,
                                          T.CATALOG_NAME,
                                          S.SCHEMA_NAME,
                                          S.ORACLE_NAME,
                                          T.SYNONYM_NAME,
                                          T.ORACLE_NAME
                                          ),
                            CURRENT_TIMESTAMP(3)
                    FROM MD_SYNONYMS T, MD_SCHEMAS S
                    WHERE S.TYPE = 'CONVERTED'
                    AND T.TYPE = S.TYPE
                    AND T.PROJECT_ID = S.PROJECT_ID
                    AND T.CONNECTION_ID = S.CONNECTION_ID
                    AND T.CATALOG_NAME = S.CATALOG_NAME
                    AND T.SCHEMA_NAME = S.SCHEMA_NAME
                    AND ((projectID IS NULL) OR (S.PROJECT_ID = projectID)) 
                    AND ((connectionID IS NULL) OR (S.CONNECTION_ID = connectionID))
                    AND (((schema_array IS NULL) AND (synonym_array IS NULL)) 
                           OR  ((schema_array IS NULL) OR (S.SCHEMA_NAME IN (SELECT COLUMN_VALUE from TABLE(schema_array)))) 
                           AND ((synonym_array IS NULL) OR (T.SYNONYM_NAME IN (SELECT COLUMN_VALUE from TABLE(synonym_array))))
                        )
                    -- AND T.CAPT_VERSION = g_captureVersionId
                    AND S.CONV_VERSION = T.CONV_VERSION
      				AND T.CONV_VERSION = g_convertVersionId  
                    );

    g_endTime := CURRENT_TIMESTAMP;
    DBMS_OUTPUT.PUT_LINE(' Migration Selected SYNONYMS Generation Time: ' || TO_CHAR(g_endTime - g_startTime));
    commit;
END generate_synonyms;

-- Generate All objects
PROCEDURE GenerateAll(projectID IN NUMBER, connectionID IN NUMBER, sourceDatabase IN VARCHAR2, convertVersion in NUMBER DEFAULT 0, password IN VARCHAR2)
IS
BEGIN

    a_startTime := CURRENT_TIMESTAMP;

    GenerateVersionID(projectID, connectionID, sourceDatabase, convertVersion);
    
--  Truncate_MD_GENERATE;
    GenerateSchemas(projectID, connectionID, sourceDatabase, password);
    GenerateTables(projectID, connectionID, sourceDatabase);
 --   GeneratePrimaryKeyConstraints(projectID, connectionID, sourceDatabase);
 --   GenerateUniqueKeyConstraints(projectID, connectionID, sourceDatabase);
 --   GenerateCheckConstraints(projectID, connectionID, sourceDatabase);
 --   GenerateIndexes(projectID, connectionID, sourceDatabase);
    
 --   GenerateSequences(projectID, connectionID, sourceDatabase);
    
    GenerateTableDefaults(projectID, connectionID, sourceDatabase);
    
 --   GenerateTriggers(projectID, connectionID, sourceDatabase);
--    GenerateViews(projectID, connectionID, sourceDatabase);
 --   GenerateStoredPrograms(projectID, connectionID, sourceDatabase);
 --   GeneratePackages(projectID, connectionID, sourceDatabase);

 --   GenerateForeignKeyConstraints(projectID, connectionID, sourceDatabase);
 --   GenerateSynonyms(projectID, connectionID, sourceDatabase);

--  GenerateUDT(projectID, connectionID, sourceDatabase);

   -- GenerateUnloadScript(unload_location_dir, my_im_role, unload_options);
  --  GenerateExternalTables(datafile_dir, logfile_dir, record_delimiter, field_terminator);
   -- GenerateOCAExternalTables( credential_name, file_uri_list, file_format);
   -- GenerateOCACopyData( credential_name, file_uri_list, file_format);

 -- -- COMMENTS --
    GenerateTableComments(projectID, connectionID, sourceDatabase);
    --GenerateViewComments(projectID, connectionID, sourceDatabase);
    GenerateColumnComments(projectID, connectionID, sourceDatabase);
    --GenerateIndexComments(projectID, connectionID, sourceDatabase);

     g_endTime := CURRENT_TIMESTAMP;
     DBMS_OUTPUT.PUT_LINE('All Objects Generation Time: ' || TO_CHAR(g_endTime - a_startTime));
END GenerateAll;

PROCEDURE Truncate_MD_GENERATE
IS
BEGIN
  EXECUTE IMMEDIATE 'TRUNCATE TABLE MD_GENERATE' ;
END Truncate_MD_GENERATE;


PROCEDURE GenerateSchemas(projectID IN NUMBER, connectionID IN NUMBER, sourceDatabase IN VARCHAR2, password IN VARCHAR2, schema_array IN generateObjectArray DEFAULT NULL ) 
IS
BEGIN
     g_startTime := CURRENT_TIMESTAMP;

	IF g_generateVersionId IS NULL or g_generateVersionId = 0 THEN
       GenerateVersionID(projectID, connectionID, sourceDatabase, 0);    
    END IF;
    
     
   INSERT INTO MD_GENERATE(
                           PROJECT_ID,
                           CONNECTION_ID,
                           CATALOG_NAME,
                           SCHEMA_NAME,
                           CAPT_VERSION,
                           CONV_VERSION,
                           GENERATE_VERSION,
                           OBJECT_TYPE,
                           OBJECT_NAME,
                           SQL_SCRIPT,
                           CREATED_ON
                          )
                   (SELECT PROJECT_ID,
                           CONNECTION_ID,
                           CATALOG_NAME,
                           ORACLE_NAME,
                           g_captureVersionId,
                           CONV_VERSION,
                           g_generateVersionId,
                           'SCHEMA',
                           ORACLE_NAME,
                           MIGRATION_GENERATOR.gen_schema_sql(
                                  PROJECT_ID,
                                  CONNECTION_ID,
                                  CATALOG_NAME,
                                  ORACLE_NAME,
                                  password
                            ),
                            CURRENT_TIMESTAMP(3)
                    FROM MD_SCHEMAS
                    WHERE 
                     TYPE = 'CONVERTED'
                     AND ((projectID IS NULL) OR (PROJECT_ID = projectID)) 
                     AND ((connectionID IS NULL) OR (CONNECTION_ID = connectionID))
                     AND ((schema_array IS NULL) OR (SCHEMA_NAME IN (SELECT COLUMN_VALUE from TABLE(schema_array))))
                  --   AND CAPT_VERSION = g_captureVersionId
                     AND CONV_VERSION = g_convertVersionId
                    );

    g_endTime := CURRENT_TIMESTAMP;
   DBMS_OUTPUT.PUT_LINE(' Migration Selected SCHEMAS Generation Time: ' || TO_CHAR(g_endTime - g_startTime));
    commit;
END GenerateSchemas;

PROCEDURE GenerateTables(projectID IN NUMBER, connectionID IN NUMBER, sourceDatabase IN VARCHAR2, schema_array IN generateObjectArray DEFAULT NULL) 
IS
BEGIN
     generate_tables(projectID, connectionID, sourceDatabase, schema_array);
END GenerateTables;

PROCEDURE GenerateTables(projectID IN NUMBER, connectionID IN NUMBER, sourceDatabase IN VARCHAR2, schema_name varchar2, table_array IN generateObjectArray DEFAULT NULL) 
IS
BEGIN
     generate_tables(projectID, connectionID, sourceDatabase, generateObjectArray(schema_name), table_array);
END GenerateTables;

PROCEDURE GenerateTableDefaults(projectID IN NUMBER, connectionID IN NUMBER, sourceDatabase IN VARCHAR2, schema_array IN generateObjectArray DEFAULT NULL) 
IS
BEGIN
     generate_table_defaults(projectID, connectionID, sourceDatabase, schema_array);
END GenerateTableDefaults;

PROCEDURE GenerateTableDefaults(projectID IN NUMBER, connectionID IN NUMBER, sourceDatabase IN VARCHAR2, schema_name varchar2, table_array IN generateObjectArray DEFAULT NULL) 
IS
BEGIN
     generate_table_defaults(projectID, connectionID, sourceDatabase, generateObjectArray(schema_name), table_array);
END GenerateTableDefaults;

PROCEDURE GeneratePrimaryKeyConstraints(projectID IN NUMBER, connectionID IN NUMBER, sourceDatabase IN VARCHAR2, schema_array IN generateObjectArray DEFAULT NULL) 
IS
BEGIN
     generate_pk_constraints(projectID, connectionID, sourceDatabase, schema_array);
END GeneratePrimaryKeyConstraints;

PROCEDURE GeneratePrimaryKeyConstraints(projectID IN NUMBER, connectionID IN NUMBER, sourceDatabase IN VARCHAR2, schema_name varchar2, table_array IN generateObjectArray DEFAULT NULL) 
IS
BEGIN
     generate_pk_constraints(projectID, connectionID, sourceDatabase, generateObjectArray(schema_name), table_array);
END GeneratePrimaryKeyConstraints;

PROCEDURE GenerateForeignKeyConstraints(projectID IN NUMBER, connectionID IN NUMBER, sourceDatabase IN VARCHAR2, schema_array IN generateObjectArray DEFAULT NULL) 
IS
BEGIN
     generate_fk_constraints(projectID, connectionID, sourceDatabase, schema_array);
END GenerateForeignKeyConstraints;

PROCEDURE GenerateForeignKeyConstraints(projectID IN NUMBER, connectionID IN NUMBER, sourceDatabase IN VARCHAR2, schema_name varchar2, table_array IN generateObjectArray DEFAULT NULL) 
IS
BEGIN
     generate_fk_constraints(projectID, connectionID, sourceDatabase, generateObjectArray(schema_name), table_array);
END GenerateForeignKeyConstraints;

PROCEDURE GenerateForeignKeyConstraints(projectID IN NUMBER, connectionID IN NUMBER, sourceDatabase IN VARCHAR2, schema_name varchar2, table_name IN varchar2, foreignKey_array IN generateObjectArray DEFAULT NULL) 
IS
BEGIN
     generate_fk_constraints(projectID, connectionID, sourceDatabase, generateObjectArray(schema_name), generateObjectArray(table_name), foreignKey_array );
END GenerateForeignKeyConstraints;


PROCEDURE GenerateUniqueKeyConstraints(projectID IN NUMBER, connectionID IN NUMBER, sourceDatabase IN VARCHAR2, schema_array IN generateObjectArray DEFAULT NULL) 
IS
BEGIN
     generate_uk_constraints(projectID, connectionID, sourceDatabase, schema_array);
END GenerateUniqueKeyConstraints;

PROCEDURE GenerateUniqueKeyConstraints(projectID IN NUMBER, connectionID IN NUMBER, sourceDatabase IN VARCHAR2, schema_name varchar2, table_array IN generateObjectArray DEFAULT NULL) 
IS
BEGIN
     generate_uk_constraints(projectID, connectionID, sourceDatabase, generateObjectArray(schema_name), table_array);
END GenerateUniqueKeyConstraints;

PROCEDURE GenerateUniqueKeyConstraints(projectID IN NUMBER, connectionID IN NUMBER, sourceDatabase IN VARCHAR2, schema_name varchar2, table_name IN varchar2, uniqueKey_array IN generateObjectArray DEFAULT NULL) 
IS
BEGIN
     generate_uk_constraints(projectID, connectionID, sourceDatabase, generateObjectArray(schema_name), generateObjectArray(table_name), uniqueKey_array );
END GenerateUniqueKeyConstraints;


PROCEDURE GenerateCheckConstraints(projectID IN NUMBER, connectionID IN NUMBER, sourceDatabase IN VARCHAR2, schema_array IN generateObjectArray DEFAULT NULL) 
IS
BEGIN
     generate_check_constraints(projectID, connectionID, sourceDatabase, schema_array);
END GenerateCheckConstraints;

PROCEDURE GenerateCheckConstraints(projectID IN NUMBER, connectionID IN NUMBER, sourceDatabase IN VARCHAR2, schema_name varchar2, table_array IN generateObjectArray DEFAULT NULL) 
IS
BEGIN
     generate_check_constraints(projectID, connectionID, sourceDatabase, generateObjectArray(schema_name), table_array);
END GenerateCheckConstraints;

PROCEDURE GenerateCheckConstraints(projectID IN NUMBER, connectionID IN NUMBER, sourceDatabase IN VARCHAR2, schema_name varchar2, table_name IN varchar2, checkConstraint_array IN generateObjectArray DEFAULT NULL) 
IS
BEGIN
     generate_check_constraints(projectID, connectionID, sourceDatabase, generateObjectArray(schema_name), generateObjectArray(table_name), checkConstraint_array );
END GenerateCheckConstraints;

PROCEDURE GenerateIndexes(projectID IN NUMBER, connectionID IN NUMBER, sourceDatabase IN VARCHAR2, schema_array IN generateObjectArray DEFAULT NULL) 
IS
BEGIN
     generate_indexes(projectID, connectionID, sourceDatabase, schema_array);
END GenerateIndexes;

PROCEDURE GenerateIndexes(projectID IN NUMBER, connectionID IN NUMBER, sourceDatabase IN VARCHAR2, schema_name varchar2, table_array IN generateObjectArray DEFAULT NULL) 
IS
BEGIN
     generate_indexes(projectID, connectionID, sourceDatabase, generateObjectArray(schema_name), table_array);
END GenerateIndexes;

PROCEDURE GenerateIndexes(projectID IN NUMBER, connectionID IN NUMBER, sourceDatabase IN VARCHAR2, schema_name varchar2, table_name IN varchar2, index_array IN generateObjectArray DEFAULT NULL) 
IS
BEGIN
     generate_indexes(projectID, connectionID, sourceDatabase, generateObjectArray(schema_name), generateObjectArray(table_name), index_array );
END GenerateIndexes;

PROCEDURE GenerateSequences(projectID IN NUMBER, connectionID IN NUMBER, sourceDatabase IN VARCHAR2, schema_array IN generateObjectArray DEFAULT NULL) 
IS
BEGIN
     generate_sequences(projectID, connectionID, sourceDatabase, schema_array);
END GenerateSequences;

PROCEDURE GenerateSequences(projectID IN NUMBER, connectionID IN NUMBER, sourceDatabase IN VARCHAR2, schema_name varchar2, sequence_array IN generateObjectArray DEFAULT NULL) 
IS
BEGIN
     generate_sequences(projectID, connectionID, sourceDatabase, generateObjectArray(schema_name), sequence_array);
END GenerateSequences;


PROCEDURE GenerateTriggers(projectID IN NUMBER, connectionID IN NUMBER, sourceDatabase IN VARCHAR2, schema_array IN generateObjectArray DEFAULT NULL) 
IS
BEGIN
     generate_triggers(projectID, connectionID, sourceDatabase, schema_array);
END GenerateTriggers;

PROCEDURE GenerateTriggers(projectID IN NUMBER, connectionID IN NUMBER, sourceDatabase IN VARCHAR2, schema_name varchar2, trigger_array IN generateObjectArray DEFAULT NULL) 
IS
BEGIN
     generate_triggers(projectID, connectionID, sourceDatabase, generateObjectArray(schema_name), trigger_array);
END GenerateTriggers;

PROCEDURE GenerateStoredPrograms(projectID IN NUMBER, connectionID IN NUMBER, sourceDatabase IN VARCHAR2, schema_array IN generateObjectArray DEFAULT NULL) 
IS
BEGIN
     generate_stored_programs(projectID, connectionID, sourceDatabase, schema_array);
END GenerateStoredPrograms;

PROCEDURE GenerateStoredPrograms(projectID IN NUMBER, connectionID IN NUMBER, sourceDatabase IN VARCHAR2, schema_name varchar2, storedProgram_array IN generateObjectArray DEFAULT NULL) 
IS
BEGIN
     generate_stored_programs(projectID, connectionID, sourceDatabase, generateObjectArray(schema_name), storedProgram_array);
END GenerateStoredPrograms;

PROCEDURE GenerateViews(projectID IN NUMBER, connectionID IN NUMBER, sourceDatabase IN VARCHAR2, schema_array IN generateObjectArray DEFAULT NULL) 
IS
BEGIN
     generate_views(projectID, connectionID, sourceDatabase, schema_array);
END GenerateViews;

PROCEDURE GenerateViews(projectID IN NUMBER, connectionID IN NUMBER, sourceDatabase IN VARCHAR2, schema_name varchar2, view_array IN generateObjectArray DEFAULT NULL) 
IS
BEGIN
     generate_views(projectID, connectionID, sourceDatabase, generateObjectArray(schema_name), view_array);
END GenerateViews;

PROCEDURE GeneratePackages(projectID IN NUMBER, connectionID IN NUMBER, sourceDatabase IN VARCHAR2, schema_array IN generateObjectArray DEFAULT NULL) 
IS
BEGIN
     generate_packages(projectID, connectionID, sourceDatabase, schema_array);
END GeneratePackages;

PROCEDURE GeneratePackages(projectID IN NUMBER, connectionID IN NUMBER, sourceDatabase IN VARCHAR2, schema_name varchar2, package_array IN generateObjectArray DEFAULT NULL) 
IS
BEGIN
     generate_packages(projectID, connectionID, sourceDatabase, generateObjectArray(schema_name), package_array);
END GeneratePackages;

PROCEDURE GenerateSynonyms(projectID IN NUMBER, connectionID IN NUMBER, sourceDatabase IN VARCHAR2, schema_array IN generateObjectArray DEFAULT NULL) 
IS
BEGIN
     generate_synonyms(projectID, connectionID, sourceDatabase, schema_array);
END GenerateSynonyms;

PROCEDURE GenerateSynonyms(projectID IN NUMBER, connectionID IN NUMBER, sourceDatabase IN VARCHAR2, schema_name varchar2, synonym_array IN generateObjectArray DEFAULT NULL) 
IS
BEGIN
     generate_synonyms(projectID, connectionID, sourceDatabase, generateObjectArray(schema_name), synonym_array);
END GenerateSynonyms;

/*
PROCEDURE GenerateUnloadScript(projectID IN NUMBER, connectionID IN NUMBER, sourceDatabase IN VARCHAR2, unload_location_dir varchar2, my_im_role varchar2, unload_options varchar2)
IS
BEGIN
    g_startTime := CURRENT_TIMESTAMP;
    
	IF g_generateVersionId IS NULL or g_generateVersionId = 0 THEN
       GenerateVersionID(projectID, connectionID, sourceDatabase, 0);    
    END IF;
    
     INSERT INTO MD_GENERATE(
                           PROJECT_ID,
                           CONNECTION_ID,
                           CATALOG_NAME,
                           SCHEMA_NAME,
                           CAPT_VERSION,
						   CONV_VERSION,
                           GENERATE_VERSION,
                           OBJECT_TYPE,
                           OBJECT_NAME,
                           SQL_SCRIPT,
                           CREATED_ON
                          )
                    (SELECT T.PROJECT_ID,
                            T.CONNECTION_ID,
                            T.CATALOG_NAME,
                            S.ORACLE_NAME,
                            g_captureVersionId,
							T.CONV_VERSION,
							g_generateVersionId,
                            'UNLOAD',
                            T.ORACLE_NAME,
                            MIGRATION_GENERATOR.generate_unload_script(
                                          T.PROJECT_ID,
                                          T.CONNECTION_ID,
                                          T.CATALOG_NAME,
                                          S.SCHEMA_NAME,
                                          S.ORACLE_NAME,
                                          T.TABLE_NAME,
                                          T.ORACLE_NAME, 
                                          unload_location_dir, 
                                          my_im_role, 
                                          unload_options
                                          ),
                            CURRENT_TIMESTAMP(3)
                    FROM MD_TABLES T, MD_SCHEMAS S
                    WHERE S.TYPE = 'CONVERTED'
                    AND T.TYPE = S.TYPE
                    AND T.PROJECT_ID = S.PROJECT_ID
                    AND T.CONNECTION_ID = S.CONNECTION_ID
                    AND T.CATALOG_NAME = S.CATALOG_NAME
                    AND T.SCHEMA_NAME = S.SCHEMA_NAME
               --   AND CAPT_VERSION = g_captureVersionId
                    AND S.CONV_VERSION = T.CONV_VERSION
                    AND T.CONV_VERSION = g_convertVersionId
                    );

    g_endTime := CURRENT_TIMESTAMP;
    DBMS_OUTPUT.PUT_LINE(' Migration Unload Scripts Generation Time: ' || TO_CHAR(g_endTime - g_startTime));
    commit;
END GenerateUnloadScript;
*/
PROCEDURE GenerateUnloadScript(projectID IN NUMBER, connectionID IN NUMBER, sourceDatabase IN VARCHAR2, unload_location_dir varchar2, aws_access_key varchar2, aws_secret_access_key varchar2, unload_options varchar2)
IS
BEGIN
    g_startTime := CURRENT_TIMESTAMP;
    
	IF g_generateVersionId IS NULL or g_generateVersionId = 0 THEN
       GenerateVersionID(projectID, connectionID, sourceDatabase, 0);    
    END IF;
    
     INSERT INTO MD_GENERATE(
                           PROJECT_ID,
                           CONNECTION_ID,
                           CATALOG_NAME,
                           SCHEMA_NAME,
                           CAPT_VERSION,
						   CONV_VERSION,
                           GENERATE_VERSION,
                           OBJECT_TYPE,
                           OBJECT_NAME,
                           SQL_SCRIPT,
                           CREATED_ON
                          )
                    (SELECT T.PROJECT_ID,
                            T.CONNECTION_ID,
                            T.CATALOG_NAME,
                            S.ORACLE_NAME,
                            g_captureVersionId,
							T.CONV_VERSION,
							g_generateVersionId,
                            'UNLOAD',
                            T.ORACLE_NAME,
                            MIGRATION_GENERATOR.generate_unload_script(
                                          T.PROJECT_ID,
                                          T.CONNECTION_ID,
                                          T.CATALOG_NAME,
                                          S.SCHEMA_NAME,
                                          S.ORACLE_NAME,
                                          T.TABLE_NAME,
                                          T.ORACLE_NAME, 
                                          unload_location_dir, 
                                          aws_access_key,
                                          aws_secret_access_key, 
                                          unload_options
                                          ),
                            CURRENT_TIMESTAMP(3)
                    FROM MD_TABLES T, MD_SCHEMAS S
                    WHERE S.TYPE = 'CONVERTED'
                    AND T.TYPE = S.TYPE
                    AND T.PROJECT_ID = S.PROJECT_ID
                    AND T.CONNECTION_ID = S.CONNECTION_ID
                    AND T.CATALOG_NAME = S.CATALOG_NAME
                    AND T.SCHEMA_NAME = S.SCHEMA_NAME
               --   AND CAPT_VERSION = g_captureVersionId
                    AND S.CONV_VERSION = T.CONV_VERSION
                    AND T.CONV_VERSION = g_convertVersionId
                    );

    g_endTime := CURRENT_TIMESTAMP;
    DBMS_OUTPUT.PUT_LINE(' Migration Unload Scripts Generation Time: ' || TO_CHAR(g_endTime - g_startTime));
    commit;
END GenerateUnloadScript;

-- Generate All External Table Script with insert into table
PROCEDURE GenerateExternalTables(projectID IN NUMBER, connectionID IN NUMBER, sourceDatabase IN VARCHAR2, datafile_dir varchar2, logfile_dir varchar2, record_delimiter varchar2, field_terminator varchar2)
IS
BEGIN
     g_startTime := CURRENT_TIMESTAMP;
       
	IF g_generateVersionId IS NULL or g_generateVersionId = 0 THEN
       GenerateVersionID(projectID, connectionID, sourceDatabase, 0);    
    END IF;

     INSERT INTO MD_GENERATE(
                           PROJECT_ID,
                           CONNECTION_ID,
                           CATALOG_NAME,
                           SCHEMA_NAME,
                           CAPT_VERSION,
						   CONV_VERSION,
                           GENERATE_VERSION,
                           OBJECT_TYPE,
                           OBJECT_NAME,
                           SQL_SCRIPT,
                           CREATED_ON
                          )
                    (SELECT T.PROJECT_ID,
                            T.CONNECTION_ID,
                            T.CATALOG_NAME,
                            S.ORACLE_NAME,
                            g_captureVersionId,
							T.CONV_VERSION,
							g_generateVersionId,
                            'EXTERNAL',
                            T.ORACLE_NAME,
                            MIGRATION_GENERATOR.generate_external_table(
                                          T.PROJECT_ID,
                                          T.CONNECTION_ID,
                                          T.CATALOG_NAME,
                                          S.SCHEMA_NAME,
                                          S.ORACLE_NAME,
                                          T.TABLE_NAME,
                                          T.ORACLE_NAME, 
                                          datafile_dir, 
                                          logfile_dir, 
                                          record_delimiter, 
                                          field_terminator
                                          ),
                            CURRENT_TIMESTAMP(3)
                    FROM MD_TABLES T, MD_SCHEMAS S
                    WHERE S.TYPE = 'CONVERTED'
                    AND T.TYPE = S.TYPE
                    AND T.PROJECT_ID = S.PROJECT_ID
                    AND T.CONNECTION_ID = S.CONNECTION_ID
                    AND T.CATALOG_NAME = S.CATALOG_NAME
                    AND T.SCHEMA_NAME = S.SCHEMA_NAME
                    --   AND CAPT_VERSION = g_captureVersionId
                    AND S.CONV_VERSION = T.CONV_VERSION
                    AND T.CONV_VERSION = g_convertVersionId
                    );

    g_endTime := CURRENT_TIMESTAMP;
    DBMS_OUTPUT.PUT_LINE(' Migration EXTERNAL TABLES Generation Time: ' || TO_CHAR(g_endTime - g_startTime));
    commit;
END GenerateExternalTables;

PROCEDURE GenerateOCAExternalTables(projectID IN NUMBER, connectionID IN NUMBER, sourceDatabase IN VARCHAR2, credential_name VARCHAR2, file_uri_list VARCHAR2, file_format VARCHAR2)
IS
BEGIN
     g_startTime := CURRENT_TIMESTAMP;
       
	IF g_generateVersionId IS NULL or g_generateVersionId = 0 THEN
       GenerateVersionID(projectID, connectionID, sourceDatabase, 0);    
    END IF;

     INSERT INTO MD_GENERATE(
                           PROJECT_ID,
                           CONNECTION_ID,
                           CATALOG_NAME,
                           SCHEMA_NAME,
                           CAPT_VERSION,
						   CONV_VERSION,
                           GENERATE_VERSION,
                           OBJECT_TYPE,
                           OBJECT_NAME,
                           SQL_SCRIPT,
                           CREATED_ON
                          )
                    (SELECT T.PROJECT_ID,
                            T.CONNECTION_ID,
                            T.CATALOG_NAME,
                            S.ORACLE_NAME,
                            g_captureVersionId,
							T.CONV_VERSION,
							g_generateVersionId,
                            'OCA_EXTERNAL',
                            T.ORACLE_NAME,
                            MIGRATION_GENERATOR.generate_OCAExternalTable(
                                          T.PROJECT_ID,
                                          T.CONNECTION_ID,
                                          T.CATALOG_NAME,
                                          S.SCHEMA_NAME,
                                          S.ORACLE_NAME,
                                          T.TABLE_NAME,
                                          T.ORACLE_NAME,
                                          credential_name,
                                          file_uri_list, 
                                          file_format
                                          ),
                            CURRENT_TIMESTAMP(3)
                    FROM MD_TABLES T, MD_SCHEMAS S
                    WHERE S.TYPE = 'CONVERTED'
                    AND T.TYPE = S.TYPE
                    AND T.PROJECT_ID = S.PROJECT_ID
                    AND T.CONNECTION_ID = S.CONNECTION_ID
                    AND T.CATALOG_NAME = S.CATALOG_NAME
                    AND T.SCHEMA_NAME = S.SCHEMA_NAME
                    --   AND CAPT_VERSION = g_captureVersionId
                    AND S.CONV_VERSION = T.CONV_VERSION
                    AND T.CONV_VERSION = g_convertVersionId
                    );

    g_endTime := CURRENT_TIMESTAMP;
    DBMS_OUTPUT.PUT_LINE(' Migration TABLES Generation Time: ' || TO_CHAR(g_endTime - g_startTime));
    commit;
END GenerateOCAExternalTables;

-- Generate DBMS_CLOUD.copy_data
PROCEDURE GenerateOCACopyData( projectID IN NUMBER, connectionID IN NUMBER, sourceDatabase IN VARCHAR2, credential_name VARCHAR2, file_uri_list VARCHAR2, file_format VARCHAR2)
IS
BEGIN
     g_startTime := CURRENT_TIMESTAMP;
       
	IF g_generateVersionId IS NULL or g_generateVersionId = 0 THEN
       GenerateVersionID(projectID, connectionID, sourceDatabase, 0);    
    END IF;


     INSERT INTO MD_GENERATE(
                           PROJECT_ID,
                           CONNECTION_ID,
                           CATALOG_NAME,
                           SCHEMA_NAME,
                           CAPT_VERSION,
						   CONV_VERSION,
                           GENERATE_VERSION,
                           OBJECT_TYPE,
                           OBJECT_NAME,
                           SQL_SCRIPT,
                           CREATED_ON
                          )
                    (SELECT T.PROJECT_ID,
                            T.CONNECTION_ID,
                            T.CATALOG_NAME,
                            S.ORACLE_NAME,
                            g_captureVersionId,
							T.CONV_VERSION,
							g_generateVersionId,
                            'OCA_COPY_DATA',
                            T.ORACLE_NAME,
                            MIGRATION_GENERATOR.generate_OCACopyData(
                                          T.PROJECT_ID,
                                          T.CONNECTION_ID,
                                          T.CATALOG_NAME,
                                          S.SCHEMA_NAME,
                                          S.ORACLE_NAME,
                                          T.TABLE_NAME,
                                          T.ORACLE_NAME,
                                          T.NUM_ROWS,
                                          g_captureVersionId,
										  T.CONV_VERSION,
							              g_generateVersionId,
                                          credential_name,
                                          file_uri_list, 
                                          file_format
                                          ),
                            CURRENT_TIMESTAMP(3)
                    FROM MD_TABLES T, MD_SCHEMAS S
                    WHERE S.TYPE = 'CONVERTED'
                    AND T.TYPE = S.TYPE
                    AND T.PROJECT_ID = S.PROJECT_ID
                    AND T.CONNECTION_ID = S.CONNECTION_ID
                    AND T.CATALOG_NAME = S.CATALOG_NAME
                    AND T.SCHEMA_NAME = S.SCHEMA_NAME
                    --   AND CAPT_VERSION = g_captureVersionId
                    AND S.CONV_VERSION = T.CONV_VERSION
                    AND T.CONV_VERSION = g_convertVersionId
                    );

    g_endTime := CURRENT_TIMESTAMP;
    DBMS_OUTPUT.PUT_LINE(' Migration TABLES Generation Time: ' || TO_CHAR(g_endTime - g_startTime));
    commit;
END GenerateOCACopyData;

-- Generate DBMS_CLOUD.copy_data using Util pkg
PROCEDURE GenerateOCACopyDataUtilPkg( projectID IN NUMBER, connectionID IN NUMBER, sourceDatabase IN VARCHAR2, credential_name VARCHAR2, file_uri_list VARCHAR2, file_format VARCHAR2)
IS
BEGIN
     g_startTime := CURRENT_TIMESTAMP;
       
	IF g_generateVersionId IS NULL or g_generateVersionId = 0 THEN
       GenerateVersionID(projectID, connectionID, sourceDatabase, 0);    
    END IF;


     INSERT INTO MD_GENERATE(
                           PROJECT_ID,
                           CONNECTION_ID,
                           CATALOG_NAME,
                           SCHEMA_NAME,
                           CAPT_VERSION,
						   CONV_VERSION,
                           GENERATE_VERSION,
                           OBJECT_TYPE,
                           OBJECT_NAME,
                           SQL_SCRIPT,
                           CREATED_ON
                          )
                    (SELECT T.PROJECT_ID,
                            T.CONNECTION_ID,
                            T.CATALOG_NAME,
                            S.ORACLE_NAME,
                            g_captureVersionId,
							T.CONV_VERSION,
							g_generateVersionId,
                            'OCA_COPY_DATA',
                            T.ORACLE_NAME,
                            MIGRATION_GENERATOR.generate_OCACopyData_utilPkg(
                                          T.PROJECT_ID,
                                          T.CONNECTION_ID,
                                          T.CATALOG_NAME,
                                          S.SCHEMA_NAME,
                                          S.ORACLE_NAME,
                                          T.TABLE_NAME,
                                          T.ORACLE_NAME,
                                          T.NUM_ROWS,
                                          g_captureVersionId,
										  T.CONV_VERSION,
							              g_generateVersionId,
                                          credential_name,
                                          file_uri_list, 
                                          file_format
                                          ),
                            CURRENT_TIMESTAMP(3)
                    FROM MD_TABLES T, MD_SCHEMAS S
                    WHERE S.TYPE = 'CONVERTED'
                    AND T.TYPE = S.TYPE
                    AND T.PROJECT_ID = S.PROJECT_ID
                    AND T.CONNECTION_ID = S.CONNECTION_ID
                    AND T.CATALOG_NAME = S.CATALOG_NAME
                    AND T.SCHEMA_NAME = S.SCHEMA_NAME
                    --   AND CAPT_VERSION = g_captureVersionId
                    AND S.CONV_VERSION = T.CONV_VERSION
                    AND T.CONV_VERSION = g_convertVersionId
                    );

    g_endTime := CURRENT_TIMESTAMP;
    DBMS_OUTPUT.PUT_LINE(' Migration TABLES Generation Time: ' || TO_CHAR(g_endTime - g_startTime));
    commit;
END GenerateOCACopyDataUtilPkg;


---
-- BEGIN OF TABLE and VIEW COMMENTS
--- 
PROCEDURE GenerateTableComments
IS
BEGIN
    g_startTime := CURRENT_TIMESTAMP;
    
    INSERT INTO MD_GENERATE(
      PROJECT_ID, 
      CONNECTION_ID, 
      CATALOG_NAME, 
      SCHEMA_NAME, 
      CAPT_VERSION,
	  CONV_VERSION,
      GENERATE_VERSION,
      OBJECT_TYPE, 
      OBJECT_NAME, 
      SQL_SCRIPT, 
      CREATED_ON
    )
    ( 
      SELECT  
         SUBCOLS.PROJECT_ID, 
         SUBCOLS.CONNECTION_ID, 
         SUBCOLS.CATALOG_NAME,  
         SUBCOLS.S_ORACLE_NAME, 
         SUBCOLS.CAPT_VERSION, 
         SUBCOLS.CONV_VERSION, 
         SUBCOLS.GENERATE_VERSION,   
         SUBCOLS.COMMENTSTRING,  
         SUBCOLS.T_ORACLE_NAME, 
         SUBCOLS.COMMENT_SCRIPT, 
         SUBCOLS.CTIME1 
      FROM 
      (    
         SELECT
            T.PROJECT_ID PROJECT_ID,     
            T.CONNECTION_ID CONNECTION_ID, 
            T.CATALOG_NAME CATALOG_NAME, 
            S.ORACLE_NAME S_ORACLE_NAME, 
            g_captureVersionId  CAPT_VERSION, 
			T.CONV_VERSION      CONV_VERSION, 
			g_generateVersionId GENERATE_VERSION, 
            'COMMENT'  COMMENTSTRING, 
            T.ORACLE_NAME T_ORACLE_NAME, 
            MIGRATION_GENERATOR.generate_table_comments(
               T.PROJECT_ID, T.CONNECTION_ID, T.CATALOG_NAME, S.SCHEMA_NAME, S.ORACLE_NAME, T.TABLE_NAME, T.ORACLE_NAME
            ) COMMENT_SCRIPT, 
            CURRENT_TIMESTAMP(3) CTIME1 
         FROM MD_TABLES T, MD_SCHEMAS S
         WHERE S.TYPE = 'CONVERTED'
           AND T.TYPE = S.TYPE
           AND T.PROJECT_ID = S.PROJECT_ID
           AND T.CONNECTION_ID = S.CONNECTION_ID
           AND T.CATALOG_NAME = S.CATALOG_NAME
           AND T.SCHEMA_NAME = S.SCHEMA_NAME  
           AND S.CONV_VERSION = T.CONV_VERSION
           AND T.CONV_VERSION = g_convertVersionId      
      ) SUBCOLS
      WHERE SUBCOLS.COMMENT_SCRIPT IS NOT NULL
    );

    g_endTime := CURRENT_TIMESTAMP;
    DBMS_OUTPUT.PUT_LINE(' Migration COMMENTS for TABLES Generation Time: ' || TO_CHAR(g_endTime - g_startTime));
    commit;
END GenerateTableComments;


PROCEDURE GenerateTableComments(projectID IN NUMBER, connectionID IN NUMBER, sourceDatabase IN VARCHAR2)
IS
BEGIN
    g_startTime := CURRENT_TIMESTAMP;

    IF g_generateVersionId IS NULL or g_generateVersionId = 0 THEN
       GenerateVersionID(projectID, connectionID, sourceDatabase, 0);    
    END IF;
    
    INSERT INTO MD_GENERATE(
      PROJECT_ID, 
      CONNECTION_ID, 
      CATALOG_NAME, 
      SCHEMA_NAME,
      CAPT_VERSION,
      CONV_VERSION,
      GENERATE_VERSION,  
      OBJECT_TYPE, 
      OBJECT_NAME, 
      SQL_SCRIPT, 
      CREATED_ON
    )
    ( 
      SELECT  
         SUBCOLS.PROJECT_ID, 
         SUBCOLS.CONNECTION_ID, 
         SUBCOLS.CATALOG_NAME, 
         SUBCOLS.S_ORACLE_NAME, 
         SUBCOLS.CAPT_VERSION,
         SUBCOLS.CONV_VERSION ,
         SUBCOLS.GENERATE_VERSION,
         SUBCOLS.TYPE_STRING,  
         SUBCOLS.T_ORACLE_NAME,
         SUBCOLS.COMMENT_SCRIPT,
         SUBCOLS.CTIME1
      FROM 
      (    
         SELECT 
            T.PROJECT_ID 						PROJECT_ID, 
            T.CONNECTION_ID 					CONNECTION_ID, 
            T.CATALOG_NAME 						CATALOG_NAME, 
            S.ORACLE_NAME 						S_ORACLE_NAME,
            g_captureVersionId  				CAPT_VERSION,
			T.CONV_VERSION      				CONV_VERSION,
			g_generateVersionId 				GENERATE_VERSION,
            'COMMENT'  							TYPE_STRING, 
            T.ORACLE_NAME 						T_ORACLE_NAME,
            MIGRATION_GENERATOR.generate_table_comments(
               T.PROJECT_ID, T.CONNECTION_ID, T.CATALOG_NAME, S.SCHEMA_NAME, S.ORACLE_NAME, T.TABLE_NAME, T.ORACLE_NAME
            ) 									COMMENT_SCRIPT,
            CURRENT_TIMESTAMP(3) 				CTIME1
         FROM MD_TABLES T, MD_SCHEMAS S
         WHERE S.TYPE = 'CONVERTED'
           AND T.TYPE = S.TYPE
           AND T.PROJECT_ID = S.PROJECT_ID
           AND T.CONNECTION_ID = S.CONNECTION_ID
           AND T.CATALOG_NAME = S.CATALOG_NAME
           AND T.SCHEMA_NAME = S.SCHEMA_NAME 
           AND S.PROJECT_ID = projectID
           AND S.CONV_VERSION = T.CONV_VERSION
           AND T.CONV_VERSION = g_convertVersionId
           AND S.CONNECTION_ID = connectionID   
      ) SUBCOLS
      WHERE SUBCOLS.COMMENT_SCRIPT IS NOT NULL
    );

    g_endTime := CURRENT_TIMESTAMP;
    DBMS_OUTPUT.PUT_LINE(' Migration Selected TABLE COMMENTS Generation Time: ' || TO_CHAR(g_endTime - g_startTime));
    commit;
END GenerateTableComments;

--
-- generate_table_comments function, returns clob
-- generates a script for the adding comments to tables.
--
FUNCTION generate_table_comments (
   projectID MD_PROJECTS.PROJECT_ID%TYPE,connectionID MD_CONNECTIONS.CONNECTION_ID%TYPE, catalogName MD_CATALOGS.CATALOG_NAME%TYPE, 
   schemaName MD_SCHEMAS.SCHEMA_NAME%TYPE, schemaOracleName MD_SCHEMAS.ORACLE_NAME%TYPE, tableName MD_TABLES.TABLE_NAME%TYPE, 
   tableOracleName MD_TABLES.ORACLE_NAME%TYPE) 
RETURN CLOB  IS

  clob_result CLOB;

  BEGIN
     SELECT TO_CLOB(tableCommentScript) into clob_result 
     FROM (
        SELECT 'COMMENT ON TABLE '|| schemaOracleName || '.' || tableOracleName ||' IS ' || '''' || T.COMMENTS || ''' '  tableCommentScript
        FROM 
           MD_TABLES T
        WHERE  T.TYPE = 'CONVERTED'
           AND T.PROJECT_ID = projectID
           AND T.CONNECTION_ID = connectionID
           AND T.CATALOG_NAME = catalogName
           AND T.SCHEMA_NAME = schemaName
           AND T.TABLE_NAME = tableName
           AND T.COMMENTS IS NOT NULL
        GROUP BY 
           T.TABLE_NAME, T.COMMENTS
     ) dual;

   -- DBMS_OUTPUT.PUT_LINE ('Comments for tables' || clob_result);
   return clob_result;

END generate_table_comments;

---
-- END OF TABLE COMMENTS
---

---
-- BEGIN OF VIEW COMMENTS
--- 
PROCEDURE GenerateViewComments
IS
BEGIN
     g_startTime := CURRENT_TIMESTAMP;

    INSERT INTO MD_GENERATE(
      PROJECT_ID, CONNECTION_ID, CATALOG_NAME, SCHEMA_NAME, OBJECT_TYPE, OBJECT_NAME, SQL_SCRIPT, CREATED_ON
    )
    ( 
      SELECT  
         SUBCOLS.PROJECT_ID, SUBCOLS.CONNECTION_ID, SUBCOLS.CATALOG_NAME, 
         SUBCOLS.S_ORACLE_NAME, 'COMMENT', SUBCOLS.V_ORACLE_NAME,
         SUBCOLS.COMMENT_SCRIPT,
         SUBCOLS.CTIME1
      FROM 
      (    
         SELECT 
            V.PROJECT_ID PROJECT_ID, V.CONNECTION_ID CONNECTION_ID, V.CATALOG_NAME CATALOG_NAME, 
            S.ORACLE_NAME S_ORACLE_NAME, 'COMMENT', V.ORACLE_NAME V_ORACLE_NAME,
            MIGRATION_GENERATOR.generate_view_comments(
               V.PROJECT_ID, V.CONNECTION_ID, V.CATALOG_NAME, S.SCHEMA_NAME, S.ORACLE_NAME, V.VIEW_NAME, V.ORACLE_NAME
            ) COMMENT_SCRIPT,
            CURRENT_TIMESTAMP(3) CTIME1
         FROM MD_VIEWS V, MD_SCHEMAS S
         WHERE S.TYPE = 'CONVERTED'
           AND V.TYPE = S.TYPE
           AND V.PROJECT_ID = S.PROJECT_ID
           AND V.CONNECTION_ID = S.CONNECTION_ID
           AND V.CATALOG_NAME = S.CATALOG_NAME
           AND V.SCHEMA_NAME = S.SCHEMA_NAME        
      ) SUBCOLS
      WHERE SUBCOLS.COMMENT_SCRIPT IS NOT NULL
    );

    g_endTime := CURRENT_TIMESTAMP;
    DBMS_OUTPUT.PUT_LINE(' Migration COMMENTS for VIEW Generation Time: ' || TO_CHAR(g_endTime - g_startTime));
    commit;
END GenerateViewComments;


PROCEDURE GenerateViewComments(projectID IN NUMBER, connectionID IN NUMBER, sourceDatabase IN VARCHAR2)
IS
BEGIN
    g_startTime := CURRENT_TIMESTAMP;

    INSERT INTO MD_GENERATE(
      PROJECT_ID, CONNECTION_ID, CATALOG_NAME, SCHEMA_NAME, OBJECT_TYPE, OBJECT_NAME, SQL_SCRIPT, CREATED_ON
    )
    ( 
      SELECT  
         SUBCOLS.PROJECT_ID, SUBCOLS.CONNECTION_ID, SUBCOLS.CATALOG_NAME, 
         SUBCOLS.S_ORACLE_NAME, 'COMMENT', SUBCOLS.V_ORACLE_NAME,
         SUBCOLS.COMMENT_SCRIPT,
         SUBCOLS.CTIME1
      FROM 
      (    
         SELECT 
            V.PROJECT_ID PROJECT_ID, V.CONNECTION_ID CONNECTION_ID, V.CATALOG_NAME CATALOG_NAME, 
            S.ORACLE_NAME S_ORACLE_NAME, 'COMMENT', V.ORACLE_NAME V_ORACLE_NAME,
            MIGRATION_GENERATOR.generate_view_comments(
               V.PROJECT_ID, V.CONNECTION_ID, V.CATALOG_NAME, S.SCHEMA_NAME, S.ORACLE_NAME, V.VIEW_NAME, V.ORACLE_NAME
            ) COMMENT_SCRIPT,
            CURRENT_TIMESTAMP(3) CTIME1
         FROM MD_VIEWS V, MD_SCHEMAS S
         WHERE S.TYPE = 'CONVERTED'
           AND V.TYPE = S.TYPE
           AND V.PROJECT_ID = S.PROJECT_ID
           AND V.CONNECTION_ID = S.CONNECTION_ID
           AND V.CATALOG_NAME = S.CATALOG_NAME
           AND V.SCHEMA_NAME = S.SCHEMA_NAME  
           AND S.PROJECT_ID = projectID
           AND S.CONNECTION_ID = connectionID      
      ) SUBCOLS
      WHERE SUBCOLS.COMMENT_SCRIPT IS NOT NULL
    );

    g_endTime := CURRENT_TIMESTAMP;
    DBMS_OUTPUT.PUT_LINE(' Migration Selected COMMENTS for VIEW Generation Time: ' || TO_CHAR(g_endTime - g_startTime));
    commit;
END GenerateViewComments;

--
-- generate_view_comments function, returns clob
-- generates a script for the adding comments to tables.
--
FUNCTION generate_view_comments (
   projectID MD_PROJECTS.PROJECT_ID%TYPE,connectionID MD_CONNECTIONS.CONNECTION_ID%TYPE, catalogName MD_CATALOGS.CATALOG_NAME%TYPE, 
   schemaName MD_SCHEMAS.SCHEMA_NAME%TYPE, schemaOracleName MD_SCHEMAS.ORACLE_NAME%TYPE, viewName MD_VIEWS.VIEW_NAME%TYPE, 
   viewOracleName MD_VIEWS.ORACLE_NAME%TYPE) 
RETURN CLOB  IS

  clob_result CLOB;

  BEGIN
     SELECT TO_CLOB(viewCommentScript) into clob_result 
     FROM (
        SELECT 'COMMENT ON TABLE '|| schemaOracleName || '.' || V.VIEW_NAME ||' IS ' || '''' || V.COMMENTS || '''' || ';' viewCommentScript
        FROM 
           MD_VIEWS V
        WHERE  V.TYPE = 'CONVERTED'
           AND V.PROJECT_ID = projectID
           AND V.CONNECTION_ID = connectionID
           AND V.CATALOG_NAME = catalogName
           AND V.SCHEMA_NAME = schemaName
           AND V.VIEW_NAME = viewName
           AND V.COMMENTS IS NOT NULL
        GROUP BY 
           V.VIEW_NAME, V.COMMENTS
     ) dual;

   -- DBMS_OUTPUT.PUT_LINE ('Comments for views ' || clob_result);
   return clob_result;

END generate_view_comments;

---
-- END OF VIEW COMMENTS
---

---
---
-- BEGIN OF INDEX COMMENTS
--- 
PROCEDURE GenerateIndexComments
IS
BEGIN
   g_startTime := CURRENT_TIMESTAMP;

   INSERT INTO MD_GENERATE(
        PROJECT_ID, CONNECTION_ID, CATALOG_NAME, SCHEMA_NAME, OBJECT_TYPE, OBJECT_NAME, SQL_SCRIPT, CREATED_ON
   )
   (
    select 
       SUBCOLS2.PROJECT_ID, SUBCOLS2.CONNECTION_ID CONNECTION_ID, 
       SUBCOLS2.CATALOG_NAME CATALOG_NAME, SUBCOLS2.S_ORACLE_NAME S_ORACLE_NAME, 
       'COMMENT', 
       SUBCOLS2.IDX_ORACLE_NAME IDX_ORACLE_NAME, 
       SUBCOLS2.COMMENT_SCRIPT,
       SUBCOLS2.CTIME1
    from 
    (
       select 
           SUBCOLS.PROJECT_ID PROJECT_ID, SUBCOLS.CONNECTION_ID CONNECTION_ID, 
           SUBCOLS.CATALOG_NAME CATALOG_NAME, SUBCOLS.S_ORACLE_NAME S_ORACLE_NAME, 
           SUBCOLS.IDX_ORACLE_NAME IDX_ORACLE_NAME, 
           MIGRATION_GENERATOR.generate_index_comments(
              SUBCOLS.PROJECT_ID, SUBCOLS.CONNECTION_ID, SUBCOLS.CATALOG_NAME, 
              SUBCOLS.S_SCHEMA_NAME, SUBCOLS.S_ORACLE_NAME, 
              SUBCOLS.IDX_INDEX_NAME, SUBCOLS.IDX_ORACLE_NAME
           ) COMMENT_SCRIPT,
           CURRENT_TIMESTAMP(3) CTIME1
       from 
       ( 
         SELECT 
            distinct IDX.PROJECT_ID PROJECT_ID, IDX.CONNECTION_ID CONNECTION_ID, 
                  IDX.CATALOG_NAME CATALOG_NAME, S.ORACLE_NAME S_ORACLE_NAME,  
                  S.SCHEMA_NAME S_SCHEMA_NAME, 
                  IDX.ORACLE_NAME IDX_ORACLE_NAME, IDX.INDEX_NAME IDX_INDEX_NAME
         FROM MD_INDEXES IDX, MD_SCHEMAS S
         WHERE S.TYPE = 'CONVERTED'
            AND IDX.TYPE = S.TYPE
            AND IDX.PROJECT_ID = S.PROJECT_ID
            AND IDX.CONNECTION_ID = S.CONNECTION_ID
            AND IDX.CATALOG_NAME = S.CATALOG_NAME
            AND IDX.SCHEMA_NAME = S.SCHEMA_NAME
      ) SUBCOLS
    ) SUBCOLS2
    WHERE SUBCOLS2.COMMENT_SCRIPT is NOT NULL
  );    

  g_endTime := CURRENT_TIMESTAMP;
  DBMS_OUTPUT.PUT_LINE(' Migration COMMENTS for INDEX Generation Time: ' || TO_CHAR(g_endTime - g_startTime));
  commit;
END GenerateIndexComments;


PROCEDURE GenerateIndexComments(projectID IN NUMBER, connectionID IN NUMBER, sourceDatabase IN VARCHAR2)
IS
BEGIN
   g_startTime := CURRENT_TIMESTAMP;

   INSERT INTO MD_GENERATE(
      PROJECT_ID, CONNECTION_ID, CATALOG_NAME, SCHEMA_NAME, OBJECT_TYPE, OBJECT_NAME, SQL_SCRIPT, CREATED_ON
   )
   (
    select 
       SUBCOLS2.PROJECT_ID, SUBCOLS2.CONNECTION_ID CONNECTION_ID, 
       SUBCOLS2.CATALOG_NAME CATALOG_NAME, SUBCOLS2.S_ORACLE_NAME S_ORACLE_NAME, 
       'COMMENT', 
       SUBCOLS2.IDX_ORACLE_NAME IDX_ORACLE_NAME, 
       SUBCOLS2.COMMENT_SCRIPT,
       SUBCOLS2.CTIME1
    from 
    (
       select 
           SUBCOLS.PROJECT_ID PROJECT_ID, SUBCOLS.CONNECTION_ID CONNECTION_ID, 
           SUBCOLS.CATALOG_NAME CATALOG_NAME, SUBCOLS.S_ORACLE_NAME S_ORACLE_NAME, 
           SUBCOLS.IDX_ORACLE_NAME IDX_ORACLE_NAME, 
           MIGRATION_GENERATOR.generate_index_comments(
              SUBCOLS.PROJECT_ID, SUBCOLS.CONNECTION_ID, SUBCOLS.CATALOG_NAME, 
              SUBCOLS.S_SCHEMA_NAME, SUBCOLS.S_ORACLE_NAME, 
              SUBCOLS.IDX_INDEX_NAME, SUBCOLS.IDX_ORACLE_NAME
           ) COMMENT_SCRIPT,
           CURRENT_TIMESTAMP(3) CTIME1
       from 
       ( 
         SELECT 
            distinct IDX.PROJECT_ID PROJECT_ID, IDX.CONNECTION_ID CONNECTION_ID, 
                  IDX.CATALOG_NAME CATALOG_NAME, S.ORACLE_NAME S_ORACLE_NAME,  
                  S.SCHEMA_NAME S_SCHEMA_NAME, 
                  IDX.ORACLE_NAME IDX_ORACLE_NAME, IDX.INDEX_NAME IDX_INDEX_NAME
         FROM MD_INDEXES IDX, MD_SCHEMAS S
         WHERE S.TYPE = 'CONVERTED'
            AND IDX.TYPE = S.TYPE
            AND IDX.PROJECT_ID = S.PROJECT_ID
            AND IDX.CONNECTION_ID = S.CONNECTION_ID
            AND IDX.CATALOG_NAME = S.CATALOG_NAME
            AND IDX.SCHEMA_NAME = S.SCHEMA_NAME
            AND S.PROJECT_ID = projectID
            AND S.CONNECTION_ID = connectionID
      ) SUBCOLS
    ) SUBCOLS2
    WHERE SUBCOLS2.COMMENT_SCRIPT is NOT NULL
  );    

  g_endTime := CURRENT_TIMESTAMP;
  DBMS_OUTPUT.PUT_LINE(' Migration Selected COMMENTS for INDEX Generation Time: ' || TO_CHAR(g_endTime - g_startTime));
  commit;
END GenerateIndexComments;

--
-- generate_index_comments function, returns clob
--
FUNCTION generate_index_comments (
   projectID MD_PROJECTS.PROJECT_ID%TYPE,connectionID MD_CONNECTIONS.CONNECTION_ID%TYPE, catalogName MD_CATALOGS.CATALOG_NAME%TYPE, 
   schemaName MD_SCHEMAS.SCHEMA_NAME%TYPE, schemaOracleName MD_SCHEMAS.ORACLE_NAME%TYPE, indexName MD_INDEXES.INDEX_NAME%TYPE, 
   indexOracleName MD_INDEXES.ORACLE_NAME%TYPE) 
RETURN CLOB  IS

  clob_result CLOB;

  BEGIN
     SELECT TO_CLOB(indexCommentScript) into clob_result 
     FROM (
        SELECT 'COMMENT ON INDEXTYPE '|| schemaOracleName || '.' || I.INDEX_NAME ||' IS ' || '''' || I.COMMENTS || '''' || ';' indexCommentScript
        FROM 
           MD_INDEXES I
        WHERE  I.TYPE = 'CONVERTED'
           AND I.PROJECT_ID = projectID
           AND I.CONNECTION_ID = connectionID
           AND I.CATALOG_NAME = catalogName
           AND I.SCHEMA_NAME = schemaName
           AND I.INDEX_NAME = indexName
           AND I.COMMENTS IS NOT NULL
        GROUP BY
           I.INDEX_NAME, I.COMMENTS
     ) dual;

   -- DBMS_OUTPUT.PUT_LINE ('Comments for indexes ' || clob_result);
   return clob_result;

END generate_index_comments;

---
-- END OF INDEX COMMENTS
---



-- BEGIN OF COLUMN COMMENTS
--- 
PROCEDURE GenerateColumnComments
IS
BEGIN
     g_startTime := CURRENT_TIMESTAMP;
     
     --IF g_generateVersionId IS NULL or g_generateVersionId = 0 THEN
     --  GenerateVersionID(projectID, connectionID, sourceDatabase, 0);    
     --END IF;

     INSERT INTO MD_GENERATE (
        PROJECT_ID, 
        CONNECTION_ID, 
        CATALOG_NAME, 
        SCHEMA_NAME, 
        CAPT_VERSION,
        CONV_VERSION,
        GENERATE_VERSION, 
        OBJECT_TYPE, 
        OBJECT_NAME, 
        SQL_SCRIPT, 
        CREATED_ON
     )   
     (
     select 
       SUBCOLS.PROJECT_ID, 
       SUBCOLS.CONNECTION_ID, 
       SUBCOLS.CATALOG_NAME, 
       SUBCOLS.S_ORACLE_NAME, 
       SUBCOLS.CAPT_VERSION,
       SUBCOLS.CONV_VERSION ,
       SUBCOLS.GENERATE_VERSION, 
       'COMMENT', 
       SUBCOLS.C_ORACLE_NAME, 
       SUBCOLS.COMMENT_SCRIPT,
       SUBCOLS.CTIME1
     from (
        select 
          PROJECT_ID, CONNECTION_ID, CATALOG_NAME, S_ORACLE_NAME, 
          g_captureVersionId    CAPT_VERSION,
		  CONV_VERSION          CONV_VERSION,
		  g_generateVersionId   GENERATE_VERSION,
          C_ORACLE_NAME, 
          MIGRATION_GENERATOR.generate_column_comments(
             PROJECT_ID, CONNECTION_ID, CATALOG_NAME, S_SCHEMA_NAME, S_ORACLE_NAME, C_COLUMN_NAME, C_ORACLE_NAME, T_TABLE_NAME
          ) comment_script, 
          CURRENT_TIMESTAMP(3) CTIME1  
        from (
           SELECT 
             distinct C.PROJECT_ID PROJECT_ID, C.CONNECTION_ID CONNECTION_ID, 
             C.CATALOG_NAME CATALOG_NAME, 
             S.ORACLE_NAME S_ORACLE_NAME, 
             C.ORACLE_NAME C_ORACLE_NAME,  
             S.SCHEMA_NAME S_SCHEMA_NAME, 
             T.TABLE_NAME T_TABLE_NAME, 
             C.COLUMN_NAME  C_COLUMN_NAME,
             T.CONV_VERSION    CONV_VERSION
           FROM MD_COLUMNS C, MD_SCHEMAS S, MD_TABLES T
           WHERE S.TYPE = 'CONVERTED'
             AND C.TYPE = S.TYPE
             AND C.PROJECT_ID = S.PROJECT_ID
             AND C.CONNECTION_ID = S.CONNECTION_ID
             AND C.CATALOG_NAME = S.CATALOG_NAME
             AND C.SCHEMA_NAME = S.SCHEMA_NAME
             AND C.TABLE_NAME = T.TABLE_NAME
             AND C.SCHEMA_NAME = T.SCHEMA_NAME
             AND C.CATALOG_NAME = T.CATALOG_NAME 
             AND S.CONV_VERSION = T.CONV_VERSION
             AND T.CONV_VERSION = g_convertVersionId  
          )
      ) SUBCOLS
      where comment_script is not NULL
    );    

    g_endTime := CURRENT_TIMESTAMP;
    DBMS_OUTPUT.PUT_LINE(' Migration COMMENTS for COLUMN Generation Time: ' || TO_CHAR(g_endTime - g_startTime));
    commit;
END GenerateColumnComments;


PROCEDURE GenerateColumnComments(projectID IN NUMBER, connectionID IN NUMBER, sourceDatabase IN VARCHAR2)
IS
BEGIN
    g_startTime := CURRENT_TIMESTAMP;

    IF g_generateVersionId IS NULL or g_generateVersionId = 0 THEN
       GenerateVersionID(projectID, connectionID, sourceDatabase, 0);    
    END IF;
     
    INSERT INTO MD_GENERATE (
       PROJECT_ID, 
        CONNECTION_ID, 
        CATALOG_NAME, 
        SCHEMA_NAME, 
        CAPT_VERSION,
        CONV_VERSION,
        GENERATE_VERSION, 
        OBJECT_TYPE, 
        OBJECT_NAME, 
        SQL_SCRIPT, 
        CREATED_ON
    )   
    (
     select 
       SUBCOLS.PROJECT_ID, 
       SUBCOLS.CONNECTION_ID, 
       SUBCOLS.CATALOG_NAME, 
       SUBCOLS.S_ORACLE_NAME, 
       SUBCOLS.CAPT_VERSION,
       SUBCOLS.CONV_VERSION ,
       SUBCOLS.GENERATE_VERSION, 
       'COMMENT', 
       SUBCOLS.C_ORACLE_NAME, 
       SUBCOLS.COMMENT_SCRIPT,
       SUBCOLS.CTIME1
     from (
        select 
          PROJECT_ID, CONNECTION_ID, CATALOG_NAME, S_ORACLE_NAME, 
          g_captureVersionId    CAPT_VERSION,
		  CONV_VERSION          CONV_VERSION,
		  g_generateVersionId   GENERATE_VERSION,
          C_ORACLE_NAME, 
          MIGRATION_GENERATOR.generate_column_comments(
             PROJECT_ID, CONNECTION_ID, CATALOG_NAME, S_SCHEMA_NAME, S_ORACLE_NAME, C_COLUMN_NAME, C_ORACLE_NAME, T_ORACLE_NAME
          ) comment_script, 
          CURRENT_TIMESTAMP(3) CTIME1  
        from (
           SELECT 
              distinct C.PROJECT_ID PROJECT_ID, C.CONNECTION_ID CONNECTION_ID, 
             C.CATALOG_NAME CATALOG_NAME, 
             S.ORACLE_NAME S_ORACLE_NAME, 
             C.ORACLE_NAME C_ORACLE_NAME,  
             S.SCHEMA_NAME S_SCHEMA_NAME, 
             T.ORACLE_NAME T_ORACLE_NAME, 
             C.COLUMN_NAME  C_COLUMN_NAME,
             T.CONV_VERSION    CONV_VERSION
           FROM MD_COLUMNS C, MD_SCHEMAS S, MD_TABLES T
           WHERE S.TYPE = 'CONVERTED'
             AND C.TYPE = S.TYPE
             AND C.PROJECT_ID = S.PROJECT_ID
             AND C.CONNECTION_ID = S.CONNECTION_ID
             AND C.CATALOG_NAME = S.CATALOG_NAME
             AND C.SCHEMA_NAME = S.SCHEMA_NAME
             AND C.TABLE_NAME = T.TABLE_NAME
             AND C.SCHEMA_NAME = T.SCHEMA_NAME
             AND C.CATALOG_NAME = T.CATALOG_NAME
             AND S.PROJECT_ID = projectID
             AND S.CONV_VERSION = T.CONV_VERSION
             AND T.CONV_VERSION = g_convertVersionId 
             AND S.CONNECTION_ID = connectionID 
          )
      ) SUBCOLS
      where comment_script is not NULL
    );    

    g_endTime := CURRENT_TIMESTAMP;
    DBMS_OUTPUT.PUT_LINE(' Migration Selected COMMENTS for COLUMN Generation Time: ' || TO_CHAR(g_endTime - g_startTime));
    commit;
END GenerateColumnComments;

--
-- generate_column_comments function, returns clob
--
FUNCTION generate_column_comments (
   projectID MD_PROJECTS.PROJECT_ID%TYPE,connectionID MD_CONNECTIONS.CONNECTION_ID%TYPE, catalogName MD_CATALOGS.CATALOG_NAME%TYPE, 
   schemaName MD_SCHEMAS.SCHEMA_NAME%TYPE, schemaOracleName MD_SCHEMAS.ORACLE_NAME%TYPE, columnName MD_COLUMNS.COLUMN_NAME%TYPE, 
   columnOracleName MD_COLUMNS.ORACLE_NAME%TYPE, tableName MD_TABLES.ORACLE_NAME%TYPE) 
RETURN CLOB  IS

  clob_result CLOB;

  BEGIN
     SELECT TO_CLOB(colCommentScript) into clob_result 
     FROM (
        SELECT 'COMMENT ON COLUMN '|| schemaOracleName || '.' || tableName || '.' ||  columnOracleName ||' IS ' || '''' || C.COMMENTS || ''' '  colCommentScript
        FROM 
           MD_COLUMNS C
        WHERE  C.TYPE = 'CONVERTED'
           AND C.PROJECT_ID = projectID
           AND C.CONNECTION_ID = connectionID
           AND C.CATALOG_NAME = catalogName
           AND C.SCHEMA_NAME = schemaName
           AND C.COLUMN_NAME = columnName
           AND C.TABLE_NAME = tableName
           AND C.COMMENTS IS NOT NULL
        GROUP BY
           C.COLUMN_NAME, C.COMMENTS
     ) dual;

   -- DBMS_OUTPUT.PUT_LINE ('Comments for columns ' || clob_result);
   return clob_result;

END generate_column_comments;

---
-- END OF COLUMNS COMMENTS
---


/* 
   DBMS_OUTPUT.PUT_LINE ('projectID Name ->' || projectID);
   DBMS_OUTPUT.PUT_LINE ('connectionID Name ->' || connectionID);
   DBMS_OUTPUT.PUT_LINE ('catalogName Name ->' || catalogName);
   DBMS_OUTPUT.PUT_LINE ('schemaName Name ->' || schemaName);
   DBMS_OUTPUT.PUT_LINE ('Table Name ->' || tableName);
   DBMS_OUTPUT.PUT_LINE ('indexName Name ->' || indexName);
*/
  END MIGRATION_GENERATOR;
  /
  