/*
 * Copyright 2005 by Oracle USA
 * 500 Oracle Parkway, Redwood Shores, California, 94065, U.S.A.
 * All rights reserved.
 */
package javax.ide.extension;

import java.net.URI;

import java.util.Collection;

import javax.ide.extension.spi.DefaultExtension;
import javax.ide.extension.spi.ExtensionSource;
import javax.ide.extension.spi.ExtensionVisitor;
import javax.ide.net.VirtualFileSystem;

/**
 * An extension hook is responsible for processing information from the
 * extension manifest for a particular feature and making this information
 * available to the service responsible for managing the functionality
 * provided by the hook.<p>
 */
public abstract class ExtensionHook extends ElementVisitor 
{
  public static final String KEY_EXTENSION = "extension";
  public static final String KEY_RSBUNDLE_CLASS = "rsbundleclass";
  public static final String ATTRIBUTE_RSBUNDLE_CLASS = "rsbundle-class";
  
  private String _schemaLocation;
  private String _providerId;
    
  /**
   * The XML namespace for a JSR-198 extension manifest.
   */
  public static final String MANIFEST_XMLNS = 
    "http://jcp.org/jsr/198/extension-manifest";
  
  /**
   * Set the id of the extension that registered this hook. This is called
   * while processing hook registration. Extension developers should 
   * generally not call this method. 
   * 
   * @param extensionId the id of the extension that registered this hook. 
   *    Must not be null.
   * @since JSR-198 2.0
   */
  public final void setProvider( String extensionId )
  {
    if ( extensionId == null )
    {
      throw new NullPointerException( "extensionId" );
    }
    if ( _providerId != null )
    {
      throw new IllegalStateException( "Provider is already set." );
    }
    _providerId = extensionId;
  }
  
  /**
   * Get the id of the extension that registered this hook.
   * 
   * @return the id of the extension that registered this hook. Generally, this
   *    is guaranteed not to be null, provided the ExtensionHook was created
   *    as part of regular hook registration processing. The "standard" JSR-198
   *    hooks will return null from this method.
   * @since JSR-198 2.0
   */
  public final String getProvider()
  {
    return _providerId;
  }
  
  /**
   * Set the location of the schema for this hook. This method is called by
   * the extension registry when processing hooks.
   * 
   * @param schemaLocation the schema location.
   */
  public final void setSchemaLocation( String schemaLocation )
  {
    _schemaLocation = schemaLocation;
  }
  
  /**
   * Get the location of the schema for this hook.
   * 
   * @return the location of the schema for this hook.
   */
  public final String getSchemaLocation()
  {
    return _schemaLocation;
  }

  /**
   * Get the resource bundle class name.
   * 
   * @param context the current context.
   * @return the fully qualified class name of the resource bundle for the
   *    current context.
   */
  protected final String getRSBundleClass( ElementContext context )
  {
    return (String) context.getScopeData().get( KEY_RSBUNDLE_CLASS );
  }

  /**
   * Get the extension that is currently being processed.
   * 
   * @param context the current context.
   * @return the extension being processed.
   */
  protected final Extension getExtension( ElementContext context )
  {
    return (Extension) context.getScopeData().get( KEY_EXTENSION );
  }
  
  /**
   * Resolves a path in the manifest. Paths depend on the source of the 
   * extension being processed. For JAR files, if the path starts with a 
   * forward slash (/), it is a path inside the JAR file. Otherwise, it is a 
   * path relative to the physical location of the JAR file.
   * 
   * @param context the xml processing context.
   * @param path a path to resolve. Must not be null.
   * @return the absolute path a resource. This may or may not exist.
   */
  protected final URI resolvePath( ElementContext context, String path )
  {
    if ( path == null )
    {
      throw new NullPointerException( "path is null" );
    }
    
    ExtensionSource source = (ExtensionSource) context.getScopeData().get( 
      ExtensionVisitor.KEY_EXTENSION_SOURCE
    );
    Extension extension = (Extension) context.getScopeData().get( 
      ExtensionHook.KEY_EXTENSION
    );
    
    return source.resolvePath( extension, path );
  }
  
  public static final class URIResource {
     private final URI m_uri;
     private final String m_extensionId;
     private URIResource(URI uri, String extensionId)
     {
         m_uri = uri;
         m_extensionId = extensionId;
     }
     
     public URI getURI() { return m_uri; }
     public String getExtensionId() { return m_extensionId; }
  }
  
  /**
   * Resolves a path in the manifest, searching all dependencies of the 
   * current extension in context until a path to an existing resource is
   * found. <p>
   * 
   * This method is significantly more expensive than 
   * {@link #resolvePath( ElementContext, String)},
   * because it must perform file system exists() checks. 
   * 
   * @param context the xml processing context.
   * @param path the path to resolve. Must not be null.
   * @return the absolute URI of a resource. If not null, this is guaranteed
   *    to have existed at the time the exists() check was made.
   * @since 2.0
   */
  protected final URIResource findPath( ElementContext context, String path )
  {
    if ( path == null )
    {
      throw new NullPointerException( "path is null" );
    }
    
    Extension extension = (Extension) context.getScopeData().get( 
        ExtensionHook.KEY_EXTENSION
    );
    
    URI def = resolvePath( context, path );
    if ( def != null && VirtualFileSystem.getVirtualFileSystem().exists( def ) )
    {
      return new URIResource(def, extension.getID());
    }
    
    // OK, start looking at deps. Sadly, this is not in any particular order,
    // per the contract of DefaultExtension.getAllImportedExtensions.
   
    if ( extension instanceof DefaultExtension )
    {
      final DefaultExtension defaultExtension = (DefaultExtension) extension;
      final Collection<String> deps = defaultExtension.getAllImportedExtensions( 
        ExtensionRegistry.getExtensionRegistry() );
      for ( String dep : deps )
      {
        final Extension e = ExtensionRegistry.getExtensionRegistry().findExtension( dep );
        if ( e != null )    // Should never happen, per getAllImportedExtensions contract, just being defensive.
        {
          if ( e instanceof DefaultExtension )
          {
            final URI u = ((DefaultExtension)e).getSource().resolvePath( e, path );
            if ( u != null && VirtualFileSystem.getVirtualFileSystem().exists( u ) )
            {
              return new URIResource(u, e.getID());
            }
          }
        }
      }
    }
    
    
    return null; 
  }
  
}
