/*
 * Copyright 2005 by Oracle USA
 * 500 Oracle Parkway, Redwood Shores, California, 94065, U.S.A.
 * All rights reserved.
 */
package javax.ide.menu;


import java.util.List;

import javax.ide.Identifiable;
import javax.ide.extension.ExtensionRegistry;
import javax.ide.menu.spi.MenuHook;
import javax.ide.util.IconDescription;
import javax.ide.view.DefaultViewable;
import javax.ide.view.Viewable;


/**
 * <code>IDEAction</code> encapsulates the meta-data describing an
 * executable action.
 */
public class IDEAction extends DefaultViewable 
  implements Identifiable, Viewable
{
  /**
   * Identifies the bound property 'mnemonic'.
   */
  public static final String PROP_MNEMONIC = "mnemonic";

  /**
   * Identifies the bound property 'checked'.
   */
  public static final String PROP_CHECKED = "checked";

  /**
   * Identifies the bound property 'enabled'.
   */
  public static final String PROP_ENABLED = "enabled";
  
  /**
   * Identifies the bound property 'toolTip'.
   */
  public static final String PROP_TOOLTIP = "toolTip";

  /**
   * Identifies the bound property 'toggleItem'.
   */
  public static final String PROP_TOGGLE_ITEM = "toggleItem";

  private String _id;
  private boolean _isToggle;
  private boolean _checked;
  private boolean _enabled = true;
  private String _label;
  private int _mnemonic;
  private String _toolTip;
  private IconDescription _icon;
  
  //---------------------------------------------------------------------------
  // Public methods.
  //---------------------------------------------------------------------------
  /**
   * Find out whether the state of this <code>IDEAction</code> is checked
   * or unchecked.
   *
   * @return The state of the <code>IDEAction</code>. 
   */
  public boolean isChecked()
  {
    return _checked;
  }

  /**
   * Set the state of the <code>IDEAction</code>.  
   * Bound property.
   *
   * @param checked The state of the <code>IDEAction</code>.
   */
  public void setChecked( boolean checked )
  {
    final Boolean oldVal = Boolean.valueOf( _checked );
    _checked = checked;
    if ( oldVal.booleanValue() != _checked )
    {
      firePropertyChange( PROP_CHECKED, oldVal, Boolean.valueOf( _checked ) );
    }
  }

  /**
   * @return <code>true</code> if the <code>IDEAction</code> is enabled.
   * <code>false</code> otherwise.
   */
  public boolean isEnabled()
  {
    return _enabled;
  }

  /**
   * Set the enabled state of the <code>IDEAction</code>.
   * Bound property.
   *
   * @param enabled The state of the <code>IDEAction</code>.
   */
  public void setEnabled( boolean enabled )
  {
    final Boolean oldVal = Boolean.valueOf( _enabled );
    _enabled = enabled;
    if ( oldVal.booleanValue() != enabled )
    {
      firePropertyChange( PROP_ENABLED, oldVal, Boolean.valueOf( _enabled ) );
    }
  }

  /**
   * Get whether this action is a toggle action.
   * 
   * @return <tt>true</tt> if this action is a toggle.
   */
  public boolean isToggleItem()
  {
    return _isToggle;
  }

  /*-
   * IDEAction interface method.
   */
  public String getLabel()
  {
    return _label;
  }
  
  /**
   * Set the action label.
   */
  public void setLabel( String label )
  {
    final String oldVal = _label;
    _label = label;
    if ( (oldVal == null && label != null) || 
         !oldVal.equals( label ) ) 
    {
      firePropertyChange( PROP_LABEL, oldVal, _label );
    }
  }

  /*-
   * IDEAction interface method.
   */
  public int getMnemonic()
  {
    return _mnemonic;
  }
  
  /**
   * Set the action label mnemonic character.
   *
   * @param mnemonic The character to be used as mnemonic.
   */
  public void setMnemonic( int mnemonic )
  {
    final Integer oldVal = new Integer( _mnemonic );
    _mnemonic = mnemonic;  
    if ( oldVal.intValue() != mnemonic )
    {
      firePropertyChange( PROP_MNEMONIC, oldVal, new Integer( _mnemonic ) );
    }
  }

  /*-
   * IDEAction interface method.
   */
  public String getToolTip()
  {
    return _toolTip;
  }

  /**
   * Set the action tooltip.
   */
  public void setToolTip( String toolTip )
  {
    final String oldVal = _toolTip;
    _toolTip = toolTip;
    if ( (oldVal == null && toolTip != null) ||
         !oldVal.equals( toolTip ) ) 
    {
      firePropertyChange( PROP_TOOLTIP, oldVal, _toolTip );
    }
  }

  /*-
   * IDEAction interface method.
   */
  public IconDescription getIcon()
  {
    return _icon;
  }

  /**
   * Set the action icon path.
   */
  public void setIcon( IconDescription icon )
  {
    final IconDescription oldVal = _icon;
    _icon = icon;
    if ( (oldVal == null && icon != null) || oldVal != icon )
    {
      firePropertyChange( PROP_ICON_PATH, oldVal, _icon );
    }
  }

  /**
   * Get the controllers for this action.
   * 
   * @return a list of controllers for this action. Must not return null.
   */
  public List /*<Controller>*/ getControllers()
  {
    MenuHook menuHook = (MenuHook)
      ExtensionRegistry.getExtensionRegistry().getHook( MenuHook.ELEMENT );
    return menuHook.getModel().getControllers( getID() );
  }

  //--------------------------------------------------------------------------
  // Object overrides.
  //--------------------------------------------------------------------------
  public String toString()
  {
    return getID().toString();
  }
  
  public int hashCode() 
  {
    return toString().hashCode();
  }

  //---------------------------------------------------------------------------
  // Methods used by IDE providers to set action property values as defined 
  // in the extension deployment descriptor.
  //---------------------------------------------------------------------------

  //--------------------------------------------------------------------------
  // Protected methods...
  //--------------------------------------------------------------------------
  /**
   * Constructor. The <code>id</code> must be a unique string 
   * identifier. This ID can be used to find the action using the
   * {@link javax.ide.menu.ActionRegistry}. 
   *
   * @param id A unique string identifying the command. The <code>id</code>
   * of an action has an undefined type, an the id name set to the string 
   * identifying the action.
   */
  public IDEAction( String id  )
  {
    _id = id;
  }
  
  public String getID()
  {
    return _id;
  }
  
  public void setToggleItem( boolean isToggleItem )
  {
    boolean oldVal = _isToggle;
    _isToggle = isToggleItem;
    if ( oldVal != isToggleItem )
    {
      firePropertyChange( PROP_TOGGLE_ITEM, 
        Boolean.valueOf( oldVal ), Boolean.valueOf( isToggleItem ));
    }
  }

}
