/*
 * Copyright (c) 2005, 2011 Oracle and/or its affiliates.
 * All rights reserved.
 */
package javax.ide.menu.spi;

import java.util.logging.Level;

import javax.ide.extension.ElementContext;
import javax.ide.extension.ElementName;
import javax.ide.extension.ElementStartContext;
import javax.ide.extension.ElementVisitor;
import javax.ide.extension.ExtensionHook;

/**
 * Visitor for menu sections.
 */
final class SectionVisitor extends PositionableVisitor
{
  final static ElementName SECTION = new ElementName(
    ExtensionHook.MANIFEST_XMLNS, "section" );

  private final static ElementName ITEM = new ElementName(
    ExtensionHook.MANIFEST_XMLNS, "item" );

  private ElementVisitor _menuVisitor = new MenuInSectionVisitor();
  private ElementVisitor _itemVisitor = new ItemVisitor();

  private final static String KEY_SECTION = "section"; // nores
  private final static String ALPHASORT = "alphasort"; // nores
  private final static String RESOLVE_MNEMONICS = "resolveMnemonics"; // nores
  private final static String DISPLAY_LABEL = "displayName"; // nores

  @Override
  protected void positionable(ElementStartContext context, String id, String before,
    String after, Float weight)
  {
    SectionContainer container = (SectionContainer)
      context.getScopeData().get( MenuHook.KEY_SECTION_CONTAINER );

    Section section = container.getSection( id );
    if ( section == null )
    {
      section = new Section( id );
      container.addSection( section );
    }

    final String alphaSort = context.getAttributeValue(ALPHASORT);
    if ( alphaSort != null )
    {
      section.setSortSectionAlphabetically( Boolean.valueOf( alphaSort ).booleanValue() );
    }

    final String resolveMnemonics = context.getAttributeValue(RESOLVE_MNEMONICS);
    if ( resolveMnemonics != null )
    {
      section.setResolveMnemonics( Boolean.valueOf( resolveMnemonics ).booleanValue() );
    }

    String displayName = context.getAttributeValue( DISPLAY_LABEL );
    if (displayName != null && ( displayName = displayName.trim() ).length() > 0 )
    {
      if ( section.getDisplayName() != null )
      {
        log( context, Level.WARNING, "The display name has already been set for section " + id );
      }
      else
      {
        section.setDisplayName( displayName );
      }
    }

    if ( weight != null  )
    {
      final Float w = section.getWeight();
      if ( w == null )
      {
        section.setWeight( weight );
      }
      else if ( !w.equals( weight ) )
      {
        log( context, Level.INFO,
             "Section " + id + " already has an assigned weight of " + w + ". " +
             "The section weight will not be redefined to " + weight + "." );
      }
    }
    else if ( before != null  )
    {
      section.setBefore( before );
    }
    else if ( after != null )
    {
      section.setAfter( after );
    }

    context.getScopeData().put( KEY_SECTION, section );

    context.registerChildVisitor( MenuVisitor.MENU, _menuVisitor );
    context.registerChildVisitor( ITEM, _itemVisitor );
  }

  private Section getSection( ElementContext context )
  {
    return (Section) context.getScopeData().get( KEY_SECTION );
  }

  private final class ItemVisitor extends PositionableVisitor
  {
    ItemVisitor()
    {
      super( "action-ref" );    // items use action-ref instead of id.
    }

    @Override
    protected void positionable(ElementStartContext context, String id,
      String before, String after, Float weight)
    {
      Section parentSection = getSection( context );
      Positionable pos = parentSection.getItem( id );

      if ( pos == null )
      {
        pos = new Item( id );
        parentSection.addItem( pos );
      }
      else
      {
        if ( pos instanceof Menu )
        {
          log( context, Level.SEVERE,
            "'"+id+"' is already defined as a menu in section '"+parentSection.getID()+"'" );
          return;
        }
      }

      Item item = (Item) pos;

      String radioGroupId = context.getAttributeValue( "radiogroup-id" );
      if ( radioGroupId != null )
      {
        radioGroupId = radioGroupId.trim();
        item.setRadioGroupID( radioGroupId );
      }

      if ( weight != null ) item.setWeight( weight );
      else if ( before != null ) item.setBefore( before );
      else if ( after != null ) item.setAfter( after );
    }
  }

  private final class MenuInSectionVisitor extends MenuVisitor
  {
    @Override
    protected void positionable(ElementStartContext context, String id,
      String before, String after, Float weight)
    {
      Section parentSection = getSection( context );

      Positionable pos = parentSection.getItem( id );

      if ( pos == null )
      {
        pos = new Menu( id );
        parentSection.addItem( pos );
      }
      else
      {
        if ( pos instanceof Item )
        {
          log( context, Level.SEVERE,
            "'"+id+"' is already defined as an item in section '"+parentSection.getID()+"'" );
          return;
        }
      }

      final Menu menu = (Menu) pos;
      processMenu( context, menu, before, after, weight );
    }
  }

}
