/*
 * @(#)JavaModel.java
 */

package javax.ide.model.java;

import javax.ide.model.java.declaration.ClassD;
import javax.ide.model.java.declaration.PackageD;
import javax.ide.model.java.declaration.TypeD;
import javax.ide.model.java.source.TreeLocator;
import javax.ide.model.java.source.TreeManager;
import javax.ide.model.java.source.TreeResolver;

/**
 * The JavaModel is the central point of access to all source
 * compilation unit facilities. Read and write access to parse trees
 * is provided through the TreeManager interface. The ability to
 * resolve source references to declarations is provided through the
 * TreeResolver interface. The ability to fetch the source for a
 * declaration is provided through the TreeLocator interface. Finally,
 * declaration loading is provided through the JavaModel interface
 * itself. A JavaModel will have an underlying classpath and
 * sourcepath that define its search path. <p/>
 *
 * @author Andy Yu
 */
public interface JavaModel
  extends TreeManager, TreeResolver, TreeLocator
{
  /**
   * Fetches a ClassD for the given fully qualified name in source format,
   * null if none. The name may denote a primitive type,
   * <code>void</code> return type, array type, or class type. <p/>
   * 
   * If multiple sources of class information are available, the most
   * up-to-date one should be returned. For example, suppose we have
   * both a class file C.class and a source file C.java that both
   * provide class information for a class C. If C.class has a more
   * recent timestamp than C.java, then a ClassD built from C.class
   * should be returned. <p/>
   *
   * Although implementations are not required to return the same
   * instance each time, they are encouraged to do so as it will
   * dramatically improve performance. <p/>
   * 
   * Note: ClassD's can be uniquely identified by their source
   * names. <p/>
   * 
   * @see TypeD#getQualifiedName()
   * 
   * @param sourceName The fully qualified name in source format, e.g.
   *                   "java.lang.Object", "java.util.Map.Entry[]".
   * 
   * @return A ClassD for the given fully qualified name, null if none can
   *         be found.
   */
  public ClassD getClass(String sourceName);

  /**
   * Fetches a TypeD for the given array type of the specified component
   * type an dimensions. <p/>
   * 
   * @param componentType The array component type.
   * @param dims The requested array dimension.
   * 
   * @return A TypeD for the type requested. If dims is zero or 
   *         componentType is null, then componentType is returned as is.
   * 
   * @throws IllegalArgumentException if dims < 0.
   */
  public TypeD getArrayType(TypeD componentType, int dims);
  
  /**
   * Fetches a TypeD for the given parameterized type. <p/>
   * 
   * @param baseType The base type for this parameterized type.
   * @param arguments Must be non-null.
   * 
   * @return A TypeD for the indicated parameterized type. If the arguments
   *         are empty, the baseType is returned as is.
   * 
   * @throws IllegalArgumentException if baseType is not a generic type or
   *                                  if the arguments do not match the
   *                                  parameters.
   */
  public TypeD getParameterizedType(TypeD baseType, TypeD[] arguments);
  
  /**
   * Fetches a PackageD for the given fully-qualified name. <p/> 
   * 
   * Although implementations are not required to return the same
   * instance each time, they are encouraged to do so as it will
   * dramatically improve performance. <p/>
   * 
   * @param sourceName The fully qualified package name. "" for the
   *                   root package.
   * 
   * @return A PackageD for the requested package.
   */
  public PackageD getPackage(String sourceName);
}
