/*
 * @(#)TreeKind.java
 */

package javax.ide.model.java.source.tree;

import java.util.Collection;
import java.util.LinkedHashMap;
import java.util.Map;

/**
 * The TreeKind enumeration identifies each individual kind of
 * Tree. <p/>
 *
 * For code-readability, all constants have been prefixed as TREE_*.
 * Constants for subinterfaces of StatementT have been prefixed as
 * TREE_STMT_*.  Constants for subinterfaces of ExpressionT have
 * been prefixed as TREE_EXPR_*.  Constants for subinterfaces of
 * DocT have been prefixed as TREE_DOC_*. <p/>
 *
 * Though the doc structure is not part of the Java Language
 * specification (moreover, there is no specification of doc
 * structure), doc comment constants have been included because
 * refactoring features may need to access and mutate doc comment
 * contents. <p/>
 *
 * In this version, this class is 1.4 compatible. In a later version,
 * it will be redone as an enum. <p/>
 *
 * @author Andy Yu
 */
public final class TreeKind
{
  private static final Map values = new LinkedHashMap();

  private static final int VALUE_base = 0;
  private static final int VALUE_ANNOTATION = VALUE_base + 0;
  private static final int VALUE_BLOCK = VALUE_base + 1;
  private static final int VALUE_CLASS_BODY = VALUE_base + 2;
  private static final int VALUE_CLASS_D = VALUE_base + 3;
  private static final int VALUE_CLASS_INITIALIZER = VALUE_base + 4;
  private static final int VALUE_CONSTRUCTOR_D = VALUE_base + 5;
  private static final int VALUE_DOC_COMMENT = VALUE_base + 6;
  private static final int VALUE_ENUM_CONSTANT_D = VALUE_base + 7;
  private static final int VALUE_FIELD_D = VALUE_base + 8;
  private static final int VALUE_FIELD_VARIABLE = VALUE_base + 9;
  private static final int VALUE_FILE = VALUE_base + 10;
  private static final int VALUE_FORMAL_PARAMETER = VALUE_base + 11;
  private static final int VALUE_FORMAL_PARAMETER_LIST = VALUE_base + 12;
  private static final int VALUE_IMPORT_D = VALUE_base + 13;
  private static final int VALUE_INTERFACES_D = VALUE_base + 14;
  private static final int VALUE_LOCAL_VARIABLE = VALUE_base + 15;
  private static final int VALUE_LOCAL_VARIABLE_D = VALUE_base + 16;
  private static final int VALUE_METHOD_D = VALUE_base + 17;
  private static final int VALUE_NAME = VALUE_base + 18;
  private static final int VALUE_PACKAGE_D = VALUE_base + 19;
  private static final int VALUE_STATEMENT_LABEL = VALUE_base + 20;
  private static final int VALUE_SUPERCLASS = VALUE_base + 21;
  private static final int VALUE_SWITCH_LABEL = VALUE_base + 22;
  private static final int VALUE_THROWS = VALUE_base + 23;
  private static final int VALUE_TYPE_REFERENCE = VALUE_base + 24;
  private static final int VALUE_TYPE_ARGUMENT = VALUE_base + 25;
  private static final int VALUE_TYPE_PARAMETER = VALUE_base + 26;
  private static final int VALUE_STMT_ASSERT = VALUE_base + 27;
  private static final int VALUE_STMT_BLOCK = VALUE_base + 28;
  private static final int VALUE_STMT_BREAK = VALUE_base + 29;
  private static final int VALUE_STMT_CATCH = VALUE_base + 30;
  private static final int VALUE_STMT_CONTINUE = VALUE_base + 31;
  private static final int VALUE_STMT_DO = VALUE_base + 32;
  private static final int VALUE_STMT_ELSE = VALUE_base + 33;
  private static final int VALUE_STMT_EMPTY = VALUE_base + 34;
  private static final int VALUE_STMT_EXPRESSION = VALUE_base + 35;
  private static final int VALUE_STMT_FINALLY = VALUE_base + 36;
  private static final int VALUE_STMT_FOR = VALUE_base + 37;
  private static final int VALUE_STMT_IF = VALUE_base + 38;
  private static final int VALUE_STMT_RETURN = VALUE_base + 39;
  private static final int VALUE_STMT_SWITCH = VALUE_base + 40;
  private static final int VALUE_STMT_SYNCH = VALUE_base + 41;
  private static final int VALUE_STMT_THROW = VALUE_base + 42;
  private static final int VALUE_STMT_TRY = VALUE_base + 43;
  private static final int VALUE_STMT_WHILE = VALUE_base + 44;
  private static final int VALUE_EXPR_ANNOTATION = VALUE_base + 45;
  private static final int VALUE_EXPR_ARRAY_ACCESS = VALUE_base + 46;
  private static final int VALUE_EXPR_ASSIGNMENT = VALUE_base + 47;
  private static final int VALUE_EXPR_DOT = VALUE_base + 48;
  private static final int VALUE_EXPR_IDENTIFIER = VALUE_base + 49;
  private static final int VALUE_EXPR_INFIX = VALUE_base + 50;
  private static final int VALUE_EXPR_LIST = VALUE_base + 51;
  private static final int VALUE_EXPR_LITERAL = VALUE_base + 52;
  private static final int VALUE_EXPR_METHOD_CALL = VALUE_base + 53;
  private static final int VALUE_EXPR_NEW_ARRAY = VALUE_base + 54;
  private static final int VALUE_EXPR_NEW_CLASS = VALUE_base + 55;
  private static final int VALUE_EXPR_QUESTION = VALUE_base + 56;
  private static final int VALUE_EXPR_TYPE = VALUE_base + 57;
  private static final int VALUE_EXPR_TYPECAST = VALUE_base + 58;
  private static final int VALUE_EXPR_UNARY = VALUE_base + 59;
  private static final int VALUE_EXPR_WRAPPER = VALUE_base + 60;
  private static final int VALUE_max = VALUE_base + 61;

  private static final int VALUE_STMT_base = VALUE_STMT_ASSERT;
  private static final int VALUE_STMT_max = VALUE_STMT_WHILE + 1;
  private static final int VALUE_EXPR_base = VALUE_EXPR_ANNOTATION;
  private static final int VALUE_EXPR_max = VALUE_EXPR_WRAPPER + 1;


  // ----------------------------------------------------------------------

  /** An annotation. */
  public static final TreeKind TREE_ANNOTATION =
    new TreeKind( VALUE_ANNOTATION, "TREE_ANNOTATION", AnnotationT.class );

  /** A code block. */
  public static final TreeKind TREE_BLOCK =
    new TreeKind( VALUE_BLOCK, "TREE_BLOCK", BlockT.class );

  /** A body of a class declaration. */
  public static final TreeKind TREE_CLASS_BODY =
    new TreeKind( VALUE_CLASS_BODY, "TREE_CLASS_BODY", ClassBodyT.class );

  /** A class, enum, interface, or annotation type declaration. */
  public static final TreeKind TREE_CLASS_D =
    new TreeKind( VALUE_CLASS_D, "TREE_CLASS_D", ClassT.class );

  /** An initializer in a class declaration. */
  public static final TreeKind TREE_CLASS_INITIALIZER =
    new TreeKind( VALUE_CLASS_INITIALIZER, "TREE_CLASS_INITIALIZER",
                  ClassInitializerT.class );

  /** A constructor declaration. */
  public static final TreeKind TREE_CONSTRUCTOR_D =
    new TreeKind( VALUE_CONSTRUCTOR_D, "TREE_CONSTRUCTOR_D", MethodT.class );

  /** A doc comment. */
  public static final TreeKind TREE_DOC_COMMENT =
    new TreeKind( VALUE_DOC_COMMENT, "TREE_DOC_COMMENT", DocCommentT.class );

  /** An enum constant declaration. */
  public static final TreeKind TREE_ENUM_CONSTANT_D =
    new TreeKind( VALUE_ENUM_CONSTANT_D, "TREE_ENUM_CONSTANT_D",
                  EnumConstantT.class );

  /** A field declaration that is not for an enum constant. */
  public static final TreeKind TREE_FIELD_D =
    new TreeKind( VALUE_FIELD_D, "TREE_FIELD_D", FieldDeclT.class );

  /** A field variable. */
  public static final TreeKind TREE_FIELD_VARIABLE =
    new TreeKind( VALUE_FIELD_VARIABLE, "TREE_FIELD_VARIABLE",
                  FieldVariableT.class );

  /** A source file (a compilation unit). */
  public static final TreeKind TREE_FILE =
    new TreeKind( VALUE_FILE, "TREE_FILE", FileT.class );

  /** A formal parameter. */
  public static final TreeKind TREE_FORMAL_PARAMETER =
    new TreeKind( VALUE_FORMAL_PARAMETER, "TREE_FORMAL_PARAMETER", 
                  FormalParameterT.class );

  /** A formal parameter list. */
  public static final TreeKind TREE_FORMAL_PARAMETER_LIST =
    new TreeKind( VALUE_FORMAL_PARAMETER_LIST, "TREE_FORMAL_PARAMETER_LIST", 
                  FormalParameterListT.class );

  /** An import declaration. */
  public static final TreeKind TREE_IMPORT_D =
    new TreeKind( VALUE_IMPORT_D, "TREE_IMPORT_D", ImportT.class );

  /** An interfaces clause for a class declaration. On a "class"
   * declaration, this will be the "implements" clause. On an
   * "interface" declaration, this will be the "extends" clause. <p/>
   * 
   * The name "interfaces" was taken from reflection. <p/>
   *
   * @see java.lang.Class#getInterfaces() */
  public static final TreeKind TREE_INTERFACES_D =
    new TreeKind( VALUE_INTERFACES_D, "TREE_INTERFACES_D", InterfacesT.class );

  /** A local variable. */
  public static final TreeKind TREE_LOCAL_VARIABLE =
    new TreeKind( VALUE_LOCAL_VARIABLE, "TREE_LOCAL_VARIABLE",
                  LocalVariableT.class );

  /** A local variable declaration. */
  public static final TreeKind TREE_LOCAL_VARIABLE_D =
    new TreeKind( VALUE_LOCAL_VARIABLE_D, "TREE_LOCAL_VARIABLE_D",
                  LocalVariableDeclT.class );

  /** A method declaration that is not for a constructor. */
  public static final TreeKind TREE_METHOD_D =
    new TreeKind( VALUE_METHOD_D, "TREE_METHOD_D", MethodT.class );

  /** A name, either a simple name or a qualified name. */
  public static final TreeKind TREE_NAME =
    new TreeKind( VALUE_NAME, "TREE_NAME", NameT.class );

  /** A package declaration. */
  public static final TreeKind TREE_PACKAGE_D =
    new TreeKind( VALUE_PACKAGE_D, "TREE_PACKAGE_D", PackageT.class );

  /** A statement label. */
  public static final TreeKind TREE_STATEMENT_LABEL =
    new TreeKind( VALUE_STATEMENT_LABEL, "TREE_STATEMENT_LABEL",
                  StatementLabelT.class );

  /** A super-class clause for a class declaration. On a "class"
   * declaration, this will be the "extends" clause. 
   *
   * The name "superclass" was taken from reflection. <p/>
   *
   * @see java.lang.Class#getSuperclass() */
  public static final TreeKind TREE_SUPERCLASS =
    new TreeKind( VALUE_SUPERCLASS, "TREE_SUPERCLASS", SuperclassT.class );
  
  /** A switch case label, either case or default. */
  public static final TreeKind TREE_SWITCH_LABEL =
    new TreeKind( VALUE_SWITCH_LABEL, "TREE_SWITCH_LABEL",
                  SwitchLabelT.class );

  /** A throws clause for a method declaration. */
  public static final TreeKind TREE_THROWS =
    new TreeKind( VALUE_THROWS, "TREE_THROWS", ThrowsT.class );

  /** A type reference. */
  public static final TreeKind TREE_TYPE_REFERENCE =
    new TreeKind( VALUE_TYPE_REFERENCE, "TREE_TYPE_REFERENCE",
                  TypeReferenceT.class );

  /** A type argument in a type reference. */
  public static final TreeKind TREE_TYPE_ARGUMENT =
    new TreeKind( VALUE_TYPE_ARGUMENT, "TREE_TYPE_ARGUMENT",
                  TypeArgumentT.class );

  /** A type parameter declaration. */
  public static final TreeKind TREE_TYPE_PARAMETER =
    new TreeKind( VALUE_TYPE_PARAMETER, "TREE_TYPE_PARAMETER",
                  TypeParameterT.class );


  // ----------------------------------------------------------------------
  // STateMenT kinds.

  /** An assert statement. */
  public static final TreeKind TREE_STMT_ASSERT =
    new TreeKind( VALUE_STMT_ASSERT, "TREE_STMT_ASSERT",
                  AssertStatementT.class );

  /** A statement wrapping a block. */
  public static final TreeKind TREE_STMT_BLOCK =
    new TreeKind( VALUE_STMT_BLOCK, "TREE_STMT_BLOCK",
                  BlockStatementT.class );

  /** A break statement. */
  public static final TreeKind TREE_STMT_BREAK =
    new TreeKind( VALUE_STMT_BREAK, "TREE_STMT_BREAK",
                  BreakStatementT.class );

  /** A catch clause in a try statement. */
  public static final TreeKind TREE_STMT_CATCH =
    new TreeKind( VALUE_STMT_CATCH, "TREE_STMT_CATCH", CatchClauseT.class );

  /** A continue statement. */
  public static final TreeKind TREE_STMT_CONTINUE =
    new TreeKind( VALUE_STMT_CONTINUE, "TREE_STMT_CONTINUE",
                  ContinueStatementT.class );

  /** A do-while statement. */
  public static final TreeKind TREE_STMT_DO =
    new TreeKind( VALUE_STMT_DO, "TREE_STMT_DO", DoStatementT.class );

  /** An else clause in an if statement. */
  public static final TreeKind TREE_STMT_ELSE =
    new TreeKind( VALUE_STMT_ELSE, "TREE_STMT_ELSE", ElseClauseT.class );

  /** An empty statement. */
  public static final TreeKind TREE_STMT_EMPTY =
    new TreeKind( VALUE_STMT_EMPTY, "TREE_STMT_EMPTY", EmptyStatementT.class );

  /** A statement wrapping an expression. */
  public static final TreeKind TREE_STMT_EXPRESSION =
    new TreeKind( VALUE_STMT_EXPRESSION, "TREE_STMT_EXPRESSION",
                  ExpressionStatementT.class );

  /** A finally clause in a try statement. */
  public static final TreeKind TREE_STMT_FINALLY =
    new TreeKind( VALUE_STMT_FINALLY, "TREE_STMT_FINALLY", 
                  FinallyClauseT.class );

  /** A for statement, all variants. */
  public static final TreeKind TREE_STMT_FOR =
    new TreeKind( VALUE_STMT_FOR, "TREE_STMT_FOR", ForStatementT.class );

  /** An if statement. */
  public static final TreeKind TREE_STMT_IF =
    new TreeKind( VALUE_STMT_IF, "TREE_STMT_IF", IfStatementT.class );

  /** A return statement. */
  public static final TreeKind TREE_STMT_RETURN =
    new TreeKind( VALUE_STMT_RETURN, "TREE_STMT_RETURN", 
                  ReturnStatementT.class );

  /** A switch statement. */
  public static final TreeKind TREE_STMT_SWITCH =
    new TreeKind( VALUE_STMT_SWITCH, "TREE_STMT_SWITCH", 
                  SwitchStatementT.class );

  /** A synchronized statement. */
  public static final TreeKind TREE_STMT_SYNCH =
    new TreeKind( VALUE_STMT_SYNCH, "TREE_STMT_SYNCH", SynchStatementT.class );

  /** A throw statement. */
  public static final TreeKind TREE_STMT_THROW =
    new TreeKind( VALUE_STMT_THROW, "TREE_STMT_THROW", ThrowStatementT.class );

  /** A try statement. */
  public static final TreeKind TREE_STMT_TRY =
    new TreeKind( VALUE_STMT_TRY, "TREE_STMT_TRY", TryStatementT.class );

  /** A while statement. */
  public static final TreeKind TREE_STMT_WHILE =
    new TreeKind( VALUE_STMT_WHILE, "TREE_STMT_WHILE", WhileStatementT.class );


  // ----------------------------------------------------------------------
  // EXPRession kinds.

  /** An expression wrapping an annotation. Used only in
   * annotations. */
  public static final TreeKind TREE_EXPR_ANNOTATION =
    new TreeKind( VALUE_EXPR_ANNOTATION, "TREE_EXPR_ANNOTATION", 
                  AnnotationExpressionT.class );

  /** An array access expression. */
  public static final TreeKind TREE_EXPR_ARRAY_ACCESS =
    new TreeKind( VALUE_EXPR_ARRAY_ACCESS, "TREE_EXPR_ARRAY_ACCESS", 
                  ArrayAccessExpressionT.class );

  /** An expression for an assignment operation. */
  public static final TreeKind TREE_EXPR_ASSIGNMENT =
    new TreeKind( VALUE_EXPR_ASSIGNMENT, "TREE_EXPR_ASSIGNMENT", 
                  AssignmentExpressionT.class );

  /** An expression for an identifier selector. */
  public static final TreeKind TREE_EXPR_DOT =
    new TreeKind( VALUE_EXPR_DOT, "TREE_EXPR_DOT", DotExpressionT.class );

  /** An identifier expression. */
  public static final TreeKind TREE_EXPR_IDENTIFIER =
    new TreeKind( VALUE_EXPR_IDENTIFIER, "TREE_EXPR_IDENTIFIER", 
                  IdentifierExpressionT.class );

  /** An expression for an infix operation. */
  public static final TreeKind TREE_EXPR_INFIX =
    new TreeKind( VALUE_EXPR_INFIX, "TREE_EXPR_INFIX", 
                  InfixExpressionT.class );

  /** A list of expressions. May be an argument list or an array
   * constant. */
  public static final TreeKind TREE_EXPR_LIST =
    new TreeKind( VALUE_EXPR_LIST, "TREE_EXPR_LIST", ListExpressionT.class );

  /** A lexer literal. Does not include class literals. */
  public static final TreeKind TREE_EXPR_LITERAL =
    new TreeKind( VALUE_EXPR_LITERAL, "TREE_EXPR_LITERAL", 
                  LiteralExpressionT.class );

  /** A method call. */
  public static final TreeKind TREE_EXPR_METHOD_CALL =
    new TreeKind( VALUE_EXPR_METHOD_CALL, "TREE_EXPR_METHOD_CALL", 
                  MethodCallExpressionT.class );

  /** An array creator expression. */
  public static final TreeKind TREE_EXPR_NEW_ARRAY =
    new TreeKind( VALUE_EXPR_NEW_ARRAY, "TREE_EXPR_NEW_ARRAY",
                  NewArrayExpressionT.class );

  /** A class creator expression. */
  public static final TreeKind TREE_EXPR_NEW_CLASS =
    new TreeKind( VALUE_EXPR_NEW_CLASS, "TREE_EXPR_NEW_CLASS", 
                  NewClassExpressionT.class );

  /** An expression for the conditional operator. */
  public static final TreeKind TREE_EXPR_QUESTION =
    new TreeKind( VALUE_EXPR_QUESTION, "TREE_EXPR_QUESTION", 
                  QuestionExpressionT.class );

  /** An expression wrapping a type reference. */
  public static final TreeKind TREE_EXPR_TYPE =
    new TreeKind( VALUE_EXPR_TYPE, "TREE_EXPR_TYPE", TypeExpressionT.class );

  /** An expression for a typecast operation. */
  public static final TreeKind TREE_EXPR_TYPECAST =
    new TreeKind( VALUE_EXPR_TYPECAST, "TREE_EXPR_TYPECAST", 
                  TypecastExpressionT.class );

  /** An expression for a prefix or postfix operation. Also
   * includes keyword selectors. */
  public static final TreeKind TREE_EXPR_UNARY =
    new TreeKind( VALUE_EXPR_UNARY, "TREE_EXPR_UNARY",
                  UnaryExpressionT.class );

  /** An expression wrapping another expression. Will be either a
   * parenthesis wrapper or a bracket wrapper. */
  public static final TreeKind TREE_EXPR_WRAPPER =
    new TreeKind( VALUE_EXPR_WRAPPER, "TREE_EXPR_WRAPPER", 
                  WrapperExpressionT.class );


  // ----------------------------------------------------------------------

  private final int ordinal;

  private final String name;

  private final Class treeClass;

  private TreeKind(int ordinal, String name, Class treeClass)
  {
    this.ordinal = ordinal;
    this.name = name;
    this.treeClass = treeClass;
    
    values.put( name, this );
  }

  public Class getTreeClass()
  {
    return treeClass;
  }

  // ----------------------------------------------------------------------

  // Begin enum compatibility section.

  public String name()
  {
    return name;
  }

  public String toString()
  {
    return name();
  }

  public int ordinal()
  {
    return ordinal;
  }

  public int hashCode()
  {
    return ordinal();
  }

  public int compareTo(TreeKind other)
  {
    return ordinal() - other.ordinal();
  }

  public boolean equals(Object other)
  {
    if (other instanceof TreeKind)
    {
      final TreeKind tk = (TreeKind) other;
      return ordinal() == tk.ordinal();
    }

    return false;
  }

  public Class getDeclaringClass()
  {
    return TreeKind.class;
  }


  // ----------------------------------------------------------------------

  public static TreeKind valueOf(int ordinal)
  {
    return values()[ ordinal ];
  }

  public static TreeKind valueOf(Class ignored, String name)
  {
    return (TreeKind) values.get(name);
  }

  public static TreeKind[] values()
  {
    final Collection entries = values.values();
    return (TreeKind[]) entries.toArray(new TreeKind[entries.size()]);
  }


  // ----------------------------------------------------------------------
}
